#include "LeafPhotosynthesis.h"
#include <math.h>

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

using namespace std;

// physical values
const double CLeafPhotosynthesis::c_dSpecificHeat = 1004.0;						// specific heat (1004 J K-1 kg -1)
const double CLeafPhotosynthesis::c_dStefanBoltzman = 5.670367 * pow(10, -8);	// (W m-2 K-4)

//  Using Ball-Berry Model
const double CLeafPhotosynthesis::c_Ganmawc = 1.647;		// rate of molecular diffusivity of water vapor

//  Using Farquhar Model
const double CLeafPhotosynthesis::c_Oi = 20.5 * 1000;		// oxygen partial pressure (Pa)
const double CLeafPhotosynthesis::c_kc = 40.4;				// Michaelis-Menten constant of Rubisco for CO2 (Pa)
const double CLeafPhotosynthesis::c_ko = 24.8 * 1000;		// Michaelis-Menten constant of Rubisco for O2 (Pa)
const double CLeafPhotosynthesis::c_f = 0.15;				// spectral correction factor	
const double CLeafPhotosynthesis::c_sl = 0.7;				// curvature of leaf response of electron transport to irradiance
const double CLeafPhotosynthesis::c_S = 710;				// electron-transport temperature response parameter (J K-1 mol-1)
const double CLeafPhotosynthesis::c_H = 220000;				// curvature parameter of Jm (J mol-1)
const double CLeafPhotosynthesis::c_R = 8.314;				// universal gas constant (J mol-1 K-1)
const double CLeafPhotosynthesis::c_dRatio_Diff_ec = 1.647;	// Molecular diffusion ratio of H2O to CO2 (Me / Mc )
const double CLeafPhotosynthesis::c_dS = 0.98;				// Co-limiting factor between Av and Aj (Collatz et al., 1991)

// activation energy, Ea based on de Pury and Farquhar (1997)
const double CLeafPhotosynthesis::c_dEa_Vl = 64800.0;
const double CLeafPhotosynthesis::c_dEa_Rl = 66400.0;
const double CLeafPhotosynthesis::c_dEa_Kc = 59400.0;
const double CLeafPhotosynthesis::c_dEa_Ko = 36000.0;

const double CLeafPhotosynthesis::c_dSaturatedRH = 100.0;	// saturated relative humidity (%)

int CLeafPhotosynthesis::m_iLeuningModel = 0;
int CLeafPhotosynthesis::m_iPhotosynthesisFunctions = 4;
int CLeafPhotosynthesis::m_iC3 = 1;
int CLeafPhotosynthesis::m_iCanopyStructure = 0;
int CLeafPhotosynthesis::m_iOmegaOptimization = 0;

double CLeafPhotosynthesis::m_dMultiLayerSplit;
double CLeafPhotosynthesis::m_dSigma_scattering;
double CLeafPhotosynthesis::m_dSoilReflectance_ppfd;
double CLeafPhotosynthesis::m_dKdd;
double CLeafPhotosynthesis::m_dOmega;
double CLeafPhotosynthesis::m_dKn;

//////////////////////////////////////////////////////////////////////
// \z/
//////////////////////////////////////////////////////////////////////

CLeafPhotosynthesis::CLeafPhotosynthesis(double dErrorValue)
{
	// set default value
	m_dBBm = 9.29;					// m : (none dimensional)
	m_dBBb = 0.008;					// b : (mol m-2 s-1)
	m_Vcmax25 = 98;					// maximum catalytic activity of Rubisco at 25oC (micro mol m-2 s-1)
	m_Jmax25 = 2.1 * m_Vcmax25;		// maximum electron transport rate at 25oC (micro mol m-2 s-1)
	m_dD0 = 0.0;

	m_dVCmax_eBL = 50;				// extended Big-Leaf Vcmax by Kosugi et al., 2005 (micro mol m-2 s-1)
	m_iIteration_eBL = 0;

	m_dErrorValue = dErrorValue;
}

CLeafPhotosynthesis::~CLeafPhotosynthesis()
{

}

void CLeafPhotosynthesis::SetParameter(const double& dVCmax_eBL)
{
	m_dVCmax_eBL = dVCmax_eBL;
}

void CLeafPhotosynthesis::SetParameter(const double& dVcmax25, const double& dJmax25, const double& dOmega, const double& dM, const double& dB)
{
	m_Vcmax25 = dVcmax25;
	m_Jmax25 = dJmax25;
	m_dOmega = dOmega;
	m_dBBm = dM;
	m_dBBb = dB;
}

void CLeafPhotosynthesis::SetParameter(const double& dVcmax25, const double& dJmax25, const double& dOmega, const double& dM, const double& dB, const double& dD0)
{
	m_Vcmax25 = dVcmax25;
	m_Jmax25 = dJmax25;
	m_dOmega = dOmega;
	m_dBBm = dM;
	m_dBBb = dB;
	m_dD0 = dD0;
}


void CLeafPhotosynthesis::Farquhar_BigLeaf_multi_layer(const double& dCO2, const double& dPPFD, const double& dAirTemperature, double dLeafTemperature, 
	const double& dVPD_air, const double& dVP_air, const double& dVP_leaf, const double& dPressure, const double& dGc, double dGb, const double& dTg, const double& dUpscalingFactor,
	double dLAI_sun, double dLAI_shade, 
	const double& dPPFD_direct, const double& dPPFD_diffuse, double dPPFD_sun, double dPPFD_shade, const double& dKb, const double& dKb2, const double& dRow_cb, const double& dRow_cd)
{
	// m_dOmega = m_dD0;
	// m_iLeuningModel = 0;

	// *******************************************************
	// additional cost using vcmax_leaf, when vcmax_leaf exceed reliably range.
	// *******************************************************
	if(m_Vcmax25 > 200.0){
		m_dAn = m_dErrorValue / 10.0;
		return;
	}

	// *******************************************************
	// nitrogen distribution factor
	// *******************************************************
	double dKn = m_dKn;
	if(m_dKn < 0.0){
		// calculating nitrogen distribution coefficient
		// Lloyd et al., 2010: Optimisation of photosynthetic carbon gain and within-canopy gradients of associated foliar traits for Amazon forest trees.
		//   Biogeosciences, 7, 1833-1859.
		dKn = exp( 0.00963 * m_Vcmax25 - 2.43 );
		if(dKn > 1.0) dKn = 1.0;
	}

	// *******************************************************
	// radiative transfer model for single layer model	
	// *******************************************************
	double dFraction_sun;
	if(m_iOmegaOptimization){
		double dLAI = dLAI_sun + dLAI_shade;
		radiative_tansfer(dPPFD_direct, dPPFD_diffuse, dLAI, dKb, dKb2, dRow_cb, dRow_cd, dKn, dPPFD_sun, dPPFD_shade, dLAI_sun, dFraction_sun);
		dLAI_shade = dLAI - dLAI_sun;
	}

	// *******************************************************
	// radiative transfer model for multi layer model	
	// *******************************************************
	vector<double> aryPPFD;					// PPFD for sunlit + shaded leaves in each layer
	vector<double> aryPPFD_sun;				// PPFD for sunlit leaf in each layer
	vector<double> aryPPFD_shade;			// PPFD for shade leaf in each layer
	vector<double> aryWeight_sun;			// weighting factor for sunlit leaf
	vector<double> aryVerticalFactor;		// vertical nitrogen distribution factor
	double dLAI_sun_layer, dLAI_shade_layer;

	int dNoUpscalingFactor = 1.0;
	double dMultiLayerSplit;
	double dLAI = dLAI_sun + dLAI_shade;
	radiative_tansfer(dPPFD_direct, dPPFD_diffuse, dPPFD_sun, dPPFD_shade, dLAI, dKb, dKb2, dRow_cb, dRow_cd, dKn, aryPPFD, aryPPFD_sun, aryPPFD_shade, aryWeight_sun, aryVerticalFactor);

	// *******************************************************
	// multi layer model for photosynthesis
	// *******************************************************
	double dGPP_sun = 0.0, dGPP_shade = 0.0, dGPP = 0.0;
	double dAv = 0.0, dAj = 0.0, dAv_sun = 0.0, dAv_shade = 0.0, dAj_sun = 0.0, dAj_shade = 0.0;
	double dPPFD_sun_layer = 0.0, dPPFD_shade_layer = 0.0;
	double dVcmax = 0.0, dJmax = 0.0, dA_gs = 0.0, dA_gs_ga = 0.0, dA_gs_ga_ts = 0.0;
	double dVcmax_sun = 0.0, dJmax_sun = 0.0, dA_gs_sun = 0.0, dA_gs_ga_sun = 0.0, dA_gs_ga_ts_sun = 0.0;
	double dVcmax_shade = 0.0, dJmax_shade = 0.0, dA_gs_shade = 0.0, dA_gs_ga_shade = 0.0, dA_gs_ga_ts_shade = 0.0;

	// relative humidity at leaf surface which is cacluated using big leaf model	
	double dQ_air  = dVP_air * 18.0 / 28.96 / (dPressure -  dVP_air);				// mixing ratio (kg kg-1)
	double dQ_leaf = dVP_leaf * 18.0 / 28.96 / (dPressure -  dVP_leaf);				// mixing ratio (kg kg-1)
	double dRhs = (dGb * dQ_air + dGc * dQ_leaf) / (dGb + dGc) / dQ_leaf;			// Su et al., 1996; Eq. 12

	double dQsurf = (dGc * dQ_leaf + dGb * dQ_air) / (dGb + dGc);			// mixing ratio (kg kg-1)
	double dVP_surf = dQsurf * dPressure / (18.0 / 28.96 + dQsurf);
	double dVPD_surf = (1.0 - dRhs) * dVP_surf;
	if(m_iLeuningModel) dRhs = 1.0 / (1.0 + dVPD_surf / m_dD0);
	if(dRhs <= 0.0) dRhs = 0.0;
	if(dRhs > 1.0) dRhs = 1.0;

	// save default parameters
	const double dVcmax25 = m_Vcmax25;
	const double dJmax25 = m_Jmax25;
	const double dB = m_dBBb;

	for(int iLayer = 0; iLayer < aryPPFD_sun.size(); iLayer++ ){
		double dWeight_sun ,dWeight_shade;

		dMultiLayerSplit = m_dMultiLayerSplit;
		if(dLAI - m_dMultiLayerSplit * iLayer < m_dMultiLayerSplit) dMultiLayerSplit = dLAI - m_dMultiLayerSplit * iLayer;

		dWeight_sun = aryWeight_sun[iLayer];
		dWeight_shade = 1.0 - dWeight_sun;
		
		if(m_iCanopyStructure == 3){
			// ********************************************
			// two leaf model in each layer
			// ********************************************
			// calculating parameters for each canopy layer
			m_Vcmax25 = dVcmax25 * aryVerticalFactor[iLayer];
			m_Jmax25 = dJmax25 * aryVerticalFactor[iLayer];
			m_dBBb = dB * aryVerticalFactor[iLayer];

			dLAI_sun_layer = dMultiLayerSplit * dWeight_sun;
			dLAI_shade_layer = dMultiLayerSplit * dWeight_shade;

			dPPFD_sun_layer = aryPPFD_sun[iLayer];
			dPPFD_shade_layer = aryPPFD_shade[iLayer];

			// sunlit leaf
			Farquhar_BigLeaf(dCO2, dPPFD_sun_layer, dAirTemperature, dLeafTemperature, dVPD_air, dVP_air, dVP_leaf, dPressure, dGc, dGb, dTg, dNoUpscalingFactor);

			if(m_dAn < 0.0){
				m_dAn = 0.0;
			}

			dGPP_sun += m_dAn * dLAI_sun_layer;
			dVcmax_sun += m_dVCmax * dLAI_sun_layer; 
			dJmax_sun += m_dJmax * dLAI_sun_layer; 
			dA_gs_sun += m_dA_gs * dLAI_sun_layer; 
			dA_gs_ga_sun += m_dA_gs_ga * dLAI_sun_layer; 
			dA_gs_ga_ts_sun += m_dA_gs_ga_ts * dLAI_sun_layer; 
			dAv_sun += m_dAv * dLAI_sun_layer;
			dAj_sun += m_dAj * dLAI_sun_layer;

			dGPP += m_dAn * dLAI_sun_layer;
			dVcmax += m_dVCmax * dLAI_sun_layer; 
			dJmax += m_dJmax * dLAI_sun_layer; 
			dA_gs += m_dA_gs * dLAI_sun_layer; 
			dA_gs_ga += m_dA_gs_ga * dLAI_sun_layer; 
			dA_gs_ga_ts += m_dA_gs_ga_ts * dLAI_sun_layer; 
			dAv += m_dAv * dLAI_sun_layer;
			dAj += m_dAj * dLAI_sun_layer;

			// shaded leaf
			Farquhar_BigLeaf(dCO2, dPPFD_shade_layer, dAirTemperature, dLeafTemperature, dVPD_air, dVP_air, dVP_leaf, dPressure, dGc, dGb, dTg, dNoUpscalingFactor);

			if(m_dAn < 0.0){
				m_dAn = 0.0;
			}

			dGPP_shade += m_dAn; // * dLAI_shade_layer;
			dVcmax_shade += m_dVCmax; // * dLAI_shade_layer;
			dJmax_shade += m_dJmax; // * dLAI_shade_layer;
			dA_gs_shade += m_dA_gs; // * dLAI_shade_layer;
			dA_gs_ga_shade += m_dA_gs_ga; // * dLAI_shade_layer;
			dA_gs_ga_ts_shade += m_dA_gs_ga_ts; // * dLAI_shade_layer;
			dAv_shade += m_dAv; // * dLAI_shade_layer;
			dAj_shade += m_dAj; // * dLAI_shade_layer;

			dGPP += m_dAn; // * dLAI_shade_layer;
			dVcmax += m_dVCmax; // * dLAI_shade_layer; 
			dJmax += m_dJmax; // * dLAI_shade_layer; 
			dA_gs += m_dA_gs; // * dLAI_shade_layer; 
			dA_gs_ga += m_dA_gs_ga; // * dLAI_shade_layer; 
			dA_gs_ga_ts += m_dA_gs_ga_ts; // * dLAI_shade_layer;
			dAv += m_dAv; // * dLAI_shade_layer;
			dAj += m_dAj; // * dLAI_shade_layer;
						
			if(m_dAn_sun == m_dErrorValue || m_dAn_shade == m_dErrorValue){
				m_dAn = m_dErrorValue;
				m_dTranspiration = m_dErrorValue;
				m_Vcmax25 = dVcmax25;
				m_Jmax25 = dJmax25;
				m_dBBb = dB;
				return;
			}
		}
		if(m_iCanopyStructure == 2){
			// ********************************************
			// big leaf model in each layer
			// ********************************************
			m_Vcmax25 = dVcmax25 * aryVerticalFactor[iLayer];
			m_Jmax25 = dJmax25 * aryVerticalFactor[iLayer];
			m_dBBb = dB * aryVerticalFactor[iLayer];

			double dPPFD_layer = aryPPFD[iLayer];
			dPPFD_sun_layer = aryPPFD_sun[iLayer];
			dPPFD_shade_layer = aryPPFD_shade[iLayer];

			Farquhar_BigLeaf(dCO2, aryPPFD[iLayer], dAirTemperature, dLeafTemperature, dVPD_air, dVP_air, dVP_leaf, dPressure, dGc, dGb, dTg, dNoUpscalingFactor);

			if(m_dAn == m_dErrorValue){
				m_dAn = m_dErrorValue;
				m_dTranspiration = m_dErrorValue;
				m_Vcmax25 = dVcmax25;
				m_Jmax25 = dJmax25;
				m_dBBb = dB;
				return;
			}

			if(m_dAn < 0.0){
				m_dAn = 0.0;
			}

			dGPP += m_dAn * dMultiLayerSplit;
			dVcmax += m_dVCmax * dMultiLayerSplit;
			dJmax += m_dJmax * dMultiLayerSplit;
			dA_gs += m_dA_gs * dMultiLayerSplit;
			dA_gs_ga += m_dA_gs_ga * dMultiLayerSplit;
			dA_gs_ga_ts += m_dA_gs_ga_ts * dMultiLayerSplit;
			dAv += m_dAv * dMultiLayerSplit;
			dAj += m_dAj * dMultiLayerSplit;
		}
	}

	m_Vcmax25 = dVcmax25;
	m_Jmax25 = dJmax25;
	m_dBBb = dB;	

	// sum of sun-leaf and shade leaf variables
	// m_dAn = dGPP_sun + dGPP_shade;
	m_dAn = dGPP;
	m_dAn_sun = dGPP_sun;
	m_dAn_shade = dGPP_shade;
	m_dVCmax = dVcmax;
	m_dJmax = dJmax;
	m_dA_gs = dA_gs;
	m_dA_gs_ga = dA_gs_ga;
	m_dA_gs_ga_ts = dA_gs_ga_ts;
	m_dQstar = m_dErrorValue;
	m_dAv = dAv;
	m_dAj = dAj;

	// ***************************************************
	// calculating gs using Ball-Berry model
	// ***************************************************
	double dRow = calculate_air_density(dLeafTemperature, dPressure, c_dSaturatedRH, 0);
	double dRow_mol  = dRow * 1000.0 / 28.96;			// air density (mol m-3) ; (28.96 g mol-1)	
	const double dRatioGb = 1.37;						// mol ratio of CO2 to H2O pow(1.6, 2/3)
	double dGbc = dGb * dRow_mol / dRatioGb;			// (m s-1 mol m-3) -> (mol m-2 s-1)
	double dGbw = dGb * dRow_mol;						// (m s-1 mol m-3) -> (mol m-2 s-1)
	double dCs = dCO2 - m_dAn / dGbc;					// CO2 concentration at leaf surface (ppm)
	m_dGsw = m_dBBm * m_dAn / dCs * dRhs + m_dBBb;		// stomatal conductance of H2O (mol m-2 s-1)	
	if(m_iLeuningModel){
		double dGanmaStar;	// CO2 compensation point in absence of mitochondrial respiration (Pa)
		double dTemperature_K = dLeafTemperature + 273.15;
		GanmaStar(dGanmaStar, dTemperature_K, dPressure);
		m_dGsw = m_dBBm * m_dAn / (dCs - dGanmaStar / (dPressure * 100.0) * pow(10, 6)) * dRhs + m_dBBb;
	}
	if(m_dGsw < m_dBBb) m_dGsw = m_dBBb;
	m_dGsc = m_dGsw / c_Ganmawc;						// stomatal conductance of CO2 (mol m-2 s-1)

	if(m_dGsc != 0.0 && dGbc != 0.0){
		m_dCi = dCO2 - m_dAn * (m_dGsc + dGbc) / m_dGsc / dGbc;
	}else{
		m_dCi = dCO2;
	}

	// ***************************************************
	// calculate transpiration 
	// ***************************************************
	double dGtw = 1.0 / (1.0 / m_dGsw + 1.0 / dGbw ) / dRow_mol;			// (m s-1)
	m_dTranspiration = dRow * dGtw * (dQ_leaf - dQ_air ) * pow(10, 6);		// mg m-2 s-1
}

void CLeafPhotosynthesis::Farquhar_BigLeaf_two_leaf(const double& dCO2, const double& dPPFD, const double& dAirTemperature, double dLeafTemperature, 
	const double& dVPD_air, const double& dVP_air, const double& dVP_leaf, const double& dPressure, const double& dGc, double dGb, const double& dTg, const double& dUpscalingFactor,
	double dPPFD_sun, double dPPFD_shade, const double& dFraction_sun, const double& dKb, const double& dLAI,
	const double& dPPFD_direct, const double& dPPFD_diffuse, const double& dKb2, const double& dRow_cb, const double& dRow_cd)
{
	// sun / shade photosynthesis by Farquhar Model based on de Pury and Farquhar (1997)
	double dGPP_sun = 0.0, dGPP_shade = 0.0;
	double dAv_sun = 0.0, dAv_shade = 0.0, dAj_sun = 0.0, dAj_shade = 0.0;
//	double dVcmax_sun, dVcmax_shade, dJmax_sun, dJmax_shade;
	double dQstar_sun, dQstar_shade;
	
	// *******************************************************
	// updated sunlit leaf fraction using Kn
	// *******************************************************
	double dSunLeafFraction = dFraction_sun, dKn;
	if(m_dKn < 0.0){
		dKn = 0.2;
		double dKn_old = 0.5;
		int iCount = 0;
		while(iCount < 10 && fabs(dKn - dKn_old) > 0.01 ){
			// in sun / shade model, dVcmax represnet Vcmax for sunlit leaf
			dSunLeafFraction = m_dOmega * ( 1.0 - exp(-dKn - dKb *  m_dOmega * dLAI) ) / (dKn + dKb *  m_dOmega * dLAI);
			dSunLeafFraction /= (1.0 - exp(-dKn) ) / dKn;
			dKn =exp( 0.00963 * (m_Vcmax25 + m_Vcmax25 * (1.0 - dSunLeafFraction) / dSunLeafFraction) / dLAI * dKn_old / ( 1.0 - exp(-dKn_old) ) - 2.43 );

			if(dKn > 5.0) break;
			iCount++;
		}
		if(dKn > 1.0 || dKn <= 0.0) dKn = 0.2;		// set realistic value
		dSunLeafFraction = m_dOmega * ( 1.0 - exp(-dKn - dKb *  m_dOmega * dLAI) ) / (dKn + dKb *  m_dOmega * dLAI);
		dSunLeafFraction /= (1.0 - exp(-dKn) ) / dKn;
	}
	
	// *******************************************************
	// radiative transfer model for single layer model	
	// *******************************************************
	if(m_iOmegaOptimization && m_iCanopyStructure == 1){
		if(m_dKn > 0.0) dKn = m_dKn;
		double dLAI_sun;
		radiative_tansfer(dPPFD_direct, dPPFD_diffuse, dLAI, dKb, dKb2, dRow_cb, dRow_cd, dKn, dPPFD_sun, dPPFD_shade, dLAI_sun, dSunLeafFraction);
	}

	// *******************************************************
	// relative humidity at leaf surface which is cacluated using big leaf model
	// *******************************************************
	double dQ_air  = dVP_air * 18.0 / 28.96 / (dPressure -  dVP_air);				// mixing ratio (kg kg-1)
	double dQ_leaf = dVP_leaf * 18.0 / 28.96 / (dPressure -  dVP_leaf);				// mixing ratio (kg kg-1)
	double dRhs = (dGb * dQ_air + dGc * dQ_leaf) / (dGb + dGc) / dQ_leaf;			// Su et al., 1996; Eq. 12

	double dQsurf = (dGc * dQ_leaf + dGb * dQ_air) / (dGb + dGc);			// mixing ratio (kg kg-1)
	double dVP_surf = dQsurf * dPressure / (18.0 / 28.96 + dQsurf);
	double dVPD_surf = (1.0 - dRhs) * dVP_surf;
	if(m_iLeuningModel) dRhs = 1.0 / (1.0 + dVPD_surf / m_dD0);
	if(dRhs <= 0.0) dRhs = 0.0;
	if(dRhs > 1.0) dRhs = 1.0;

	// save default parameters
	const double dVcmax25 = m_Vcmax25;
	const double dJmax25 = m_Jmax25;
	const double dB = m_dBBb;
	double dNoUpscalingFactor = 1.0;

	// *******************************************************
	// sun leaf	
	// *******************************************************
	m_Vcmax25 = dVcmax25 * dSunLeafFraction;
	m_Jmax25 = dJmax25 * dSunLeafFraction;
	m_dBBb = dB * dSunLeafFraction;
	Farquhar_BigLeaf(dCO2, dPPFD_sun, dAirTemperature, dLeafTemperature, dVPD_air, dVP_air, dVP_leaf, dPressure, dGc, dGb, dTg, dNoUpscalingFactor);
	dGPP_sun = m_dAn;
	m_dAn_sun = m_dAn;
	m_dRd_sun = m_dRd;
	m_dVCmax_sun = m_dVCmax;
	m_dJmax_sun = m_dJmax;
	m_dA_gs_sun = m_dA_gs;
	m_dA_gs_ga_sun = m_dA_gs_ga;
	m_dA_gs_ga_ts_sun = m_dA_gs_ga_ts;
	dQstar_sun = m_dQstar;
	dAv_sun = m_dAv;
	dAj_sun = m_dAj;

	if(m_dAn == m_dErrorValue){
		m_Vcmax25 = dVcmax25;
		m_Jmax25 = dJmax25;
		m_dBBb = dB;
		return;
	}

	// *******************************************************
	// shade leaf
	// *******************************************************
	// m_Vcmax25 = dVcmax25 * dFraction_shade / dFraction_sun;
	// m_Jmax25 = dJmax25 * dFraction_shade / dFraction_sun;
	// m_dBBb = dB * dFraction_shade / dFraction_sun;
	// m_Vcmax25 = dVcmax25 / dSunLeafFraction - dVcmax25;
	// m_Jmax25 = dJmax25 / dSunLeafFraction - dJmax25;
	// m_dBBb = dB / dSunLeafFraction - dB;
	m_Vcmax25 = dVcmax25 * (1.0 - dSunLeafFraction);
	m_Jmax25 = dJmax25 * (1.0 - dSunLeafFraction);
	m_dBBb = dB * (1.0 - dSunLeafFraction);
	Farquhar_BigLeaf(dCO2, dPPFD_shade, dAirTemperature, dLeafTemperature, dVPD_air, dVP_air, dVP_leaf, dPressure, dGc, dGb, dTg, dNoUpscalingFactor);
	dGPP_shade = m_dAn;
	m_dAn_shade = m_dAn;
	m_dRd_shade = m_dRd;
	m_dVCmax_shade = m_dVCmax;
	m_dJmax_shade = m_dJmax;
	m_dA_gs_shade = m_dA_gs;
	m_dA_gs_ga_shade = m_dA_gs_ga;
	m_dA_gs_ga_ts_shade = m_dA_gs_ga_ts;
	dQstar_shade = m_dQstar + m_dQstar;
	dAv_shade = m_dAv;
	dAj_shade = m_dAj;

	if(m_dAn == m_dErrorValue){
		m_Vcmax25 = dVcmax25;
		m_Jmax25 = dJmax25;
		m_dBBb = dB;
		return;
	}

	// sum of sun-leaf and shade leaf variables
	m_dAn = dGPP_sun + dGPP_shade;
	m_dRd = m_dRd_sun + m_dRd_shade;
	m_dVCmax = m_dVCmax_sun + m_dVCmax_shade;
	m_dJmax = m_dJmax_sun + m_dJmax_shade;
	m_dA_gs = m_dA_gs_sun + m_dA_gs_shade;
	m_dA_gs_ga = m_dA_gs_ga_sun + m_dA_gs_ga_shade;
	m_dA_gs_ga_ts = m_dA_gs_ga_ts_sun + m_dA_gs_ga_ts_shade;
	// m_dQstar = dQstar_sun + dQstar_shade;
	m_dQstar = dQstar_shade;
	m_dAv = dAv_sun + dAv_shade;
	m_dAj = dAj_sun + dAj_shade;

	m_Vcmax25 = dVcmax25;
	m_Jmax25 = dJmax25;
	m_dBBb = dB;

	// ***************************************************
	// calculating gs using Ball-Berry model
	// ***************************************************
	double dRow = calculate_air_density(dLeafTemperature, dPressure, c_dSaturatedRH, 0);
	double dRow_mol  = dRow * 1000.0 / 28.96;			// air density (mol m-3) ; (28.96 g mol-1)	
	const double dRatioGb = 1.37;						// mol ratio of CO2 to H2O pow(1.6, 2/3)
	double dGbc = dGb * dRow_mol / dRatioGb;			// (m s-1 mol m-3) -> (mol m-2 s-1)
	double dGbw = dGb * dRow_mol;						// (m s-1 mol m-3) -> (mol m-2 s-1)
	double dCs = dCO2 - m_dAn / dGbc;					// CO2 concentration at leaf surface (ppm)
	m_dGsw = m_dBBm * m_dAn / dCs * dRhs + m_dBBb;		// stomatal conductance of H2O (mol m-2 s-1)
	if(m_iLeuningModel){
		double dGanmaStar;
		double dTemperature_K = dLeafTemperature + 273.15;
		GanmaStar(dGanmaStar, dTemperature_K, dPressure);
		m_dGsw = m_dBBm * m_dAn / (dCs - dGanmaStar / (dPressure * 100.0) * pow(10, 6)) * dRhs + m_dBBb;
	}
	if(m_dGsw < m_dBBb) m_dGsw = m_dBBb;
	m_dGsc = m_dGsw / c_Ganmawc;						// stomatal conductance of CO2 (mol m-2 s-1)

	if(m_dGsc != 0.0 && dGbc != 0.0){
		m_dCi = dCO2 - m_dAn * (m_dGsc + dGbc) / m_dGsc / dGbc;
	}else{
		m_dCi = dCO2;
	}

	// ***************************************************
	// calculate transpiration 
	// ***************************************************
	double dGtw = 1.0 / (1.0 / m_dGsw + 1.0 / dGbw ) / dRow_mol;			// (m s-1)
	m_dTranspiration = dRow * dGtw * (dQ_leaf - dQ_air ) * pow(10, 6);		// mg m-2 s-1
}

void CLeafPhotosynthesis::Farquhar_BigLeaf_coupled_EB(const double& dCO2, double dPPFD, const double& dAirTemperature, double& dLeafTemperature, 
	const double& dVPD_air, const double& dVP_air, const double& dVP_leaf, const double& dPressure, const double& dGc, double dGb,
	const double& dH, const double& dLE, const double& dG, const double& dT_air, const double& dTg, const double& dRamda, const double& dUpscalingFactor,
	const double& dLAI, const double& dPPFD_direct, const double& dPPFD_diffuse, const double& dKb, const double& dKb2, const double& dRow_cb, const double& dRow_cd)
{
	// dCO2					: atmospheric CO2 concentration (ppm)
	// dPPFD				: photosynthetically active radiation (umol m-2 s-1)
	// dTemperature			: leaf temperature (degree C)
	// dVPD_air				: atmospheric vapor pressure deficit (hPa)
	// dVP_air				: atmospheric vapor pressure (hPa)
	// dVP_leaf				: leaf vapor pressure (hPa)
	// dPressure			: atmospheric pressure (hPa)
	// dGc					: surface layer conductance condutance (m s-1)
	// dGb					: boudary layer conductance condutance of H2O (m s-1)
	// dH					: sensible heat flux (W m-2)
	// dG					: ground heat flux (W m-2)
	// dT_air				: air temperature (degree C)
	// dTg                  : growth temperature (degree C)
	// dRamda				: latent heat of water vaporization (J kg-1)
	// dUpscalingFactor     : upscaling factor of resiration from leaf scale to canopy scale (m2 m-2)

	// *******************************************************
	// radiative transfer model for single layer model	
	// *******************************************************
	if(m_iOmegaOptimization && m_iCanopyStructure == 0){
		double dLAI_sun, dFraction_sun, dKn = 0.2, dPPFD_sun, dPPFD_shade;
		radiative_tansfer(dPPFD_direct, dPPFD_diffuse, dLAI, dKb, dKb2, dRow_cb, dRow_cd, dKn, dPPFD_sun, dPPFD_shade, dLAI_sun, dFraction_sun);
		dPPFD = dPPFD_sun + dPPFD_shade;
	}

	// *******************************************************
	// Big-Leaf model
	// *******************************************************
	double dRnet_star;					// net radiation minus outgoing longwave raditaion (W m-2)
	double dLE_model = dLE;				// latent heat flux by model (inital value is substituted by observation)
	double dT_leaf = dLeafTemperature;	// degree C
	double dT_leaf_old = m_dErrorValue;	// degree C
	double dVP_leaf_new = dVP_leaf;
	double dRlup;						// upward longwave radiation
	double dRow;						// air density (kg m-3)
	int iItteration = 0;
	const int cItteration_max = 10;

	dRnet_star = dH + dLE_model + dG + c_dStefanBoltzman * pow(dT_leaf + 273.15, 4);
	// dRow = 1.293 / (1.0 + 0.00367 * dT_leaf );
	dRow  = calculate_air_density(dT_leaf, dPressure, c_dSaturatedRH, 0);

	while(iItteration < cItteration_max){
		Farquhar_BigLeaf(dCO2, dPPFD, dAirTemperature, dT_leaf, dVPD_air, dVP_air, dVP_leaf_new, dPressure, dGc, dGb, dTg, dUpscalingFactor);

		dRlup = c_dStefanBoltzman * pow(dT_leaf + 273.15, 4);
		dLE_model = m_dTranspiration * dRamda / pow(10, 6);
		if(m_dTranspiration == m_dErrorValue){
			break;
		}
		dT_leaf = (dRnet_star - dRlup - dLE_model - dG) / dGb / dRow / c_dSpecificHeat + dT_air;
		if( fabs(dT_leaf - dT_leaf_old) > 0.1){
			dT_leaf_old = dT_leaf;
			dVP_leaf_new = Goff_Gratch(dT_leaf, dPressure);
			iItteration++;
		}
		else{
			dLeafTemperature = dT_leaf;
			break;
		}
	}
}

void CLeafPhotosynthesis::Farquhar_BigLeaf(const double& dCO2, double dPPFD, const double& dAirTemperature, double dLeafTemperature, 
	const double& dVPD_air, const double& dVP_air, const double& dVP_leaf, const double& dPressure, const double& dGc, double dGb, const double& dTg, const double& dUpscalingFactor,
	const double& dLAI, const double& dPPFD_direct, const double& dPPFD_diffuse, const double& dKb, const double& dKb2, const double& dRow_cb, const double& dRow_cd)
{	
	// *******************************************************
	// radiative transfer model for single layer model	
	// *******************************************************
	if(m_iOmegaOptimization && m_iCanopyStructure == 0){
		double dLAI_sun, dFraction_sun, dKn = 0.2, dPPFD_sun, dPPFD_shade;
		radiative_tansfer(dPPFD_direct, dPPFD_diffuse, dLAI, dKb, dKb2, dRow_cb, dRow_cd, dKn, dPPFD_sun, dPPFD_shade, dLAI_sun, dFraction_sun);
		dPPFD = dPPFD_sun + dPPFD_shade;
	}

	// *******************************************************
	// Big-Leaf model
	// *******************************************************
	Farquhar_BigLeaf(dCO2, dPPFD,dAirTemperature,dLeafTemperature, dVPD_air,dVP_air, dVP_leaf, dPressure, dGc, dGb, dTg, dUpscalingFactor);
}

void CLeafPhotosynthesis::Farquhar_BigLeaf(const double& dCO2, double dPPFD, const double& dAirTemperature, double dLeafTemperature, 
	const double& dVPD_air, const double& dVP_air, const double& dVP_leaf, const double& dPressure, const double& dGc, double dGb, const double& dTg, const double& dUpscalingFactor)
{
	// leaf photosynthesis by Farquhar Model based on de Pury and Farquhar (1997)
	// dCO2					: atmospheric CO2 concentration (ppm)
	// dPPFD				: photosynthetically active radiation (umol m-2 s-1)
	// dLeafTemperature		: leaf temperature (degree C)
	// dVPD_air				: atmospheric vapor pressure deficit (hPa)
	// dVP_air				: atmospheric vapor pressure (hPa)
	// dVP_leaf				: leaf vapor pressure (hPa)
	// dPressure			: atmospheric pressure (hPa)
	// dGc					: surface layer conductance condutance (m s-1)
	// dGb					: boudary layer conductance condutance of H2O (m s-1)
	// dTg                  : growth temperature (degreeC)
	// dUpscalingFactor     : upscaling factor of resiration from leaf scale to canopy scale (m2 m-2)

	double dJm;				//  potential rate of electron transport rate per unit leaf area (umol m-2 s-1)
	double dKo;				//  Michaelis-Menten const of Rubisco for O2 (Pa)
	double dKc;				//  Michaelis-Menten const of Rubisco for CO2 (Pa)

	double dAv;				//  Rubisco-limited photosynthesis (umol m-2 s-1)
	double dAj;				//  Electron-transport limited rate of photosynthesis (umol m-2 s-1)
	double dRd;				// leaf respiration per unit leaf area (umol m-2 s-1)
	double dJ;				//  rate of electron transport rate per unit leaf area (umol m-2 s-1)
	double dVCmax;			//  photosynthesis Rubisco canopy per unit leaf area (umol m-2 s-1)
	double dKp;				//  Effective Michaelis-Menten constant of Rubisco (Pa)
	double dGanmaStar;		//   in the absence of mitochondria respiraton (Pa)
	double dGbc;			// boudary layer conductance of CO2 (mol m-2 s-1)
	double dGbw;			// boudary layer conductance of H2O (mol m-2 s-1)

	double dVCmax_air;		//  photosynthesis Rubisco canopy per unit leaf area at air temperature(umol m-2 s-1)
	double dRd_air;			// leaf respiration per unit leaf area at air temperature (umol m-2 s-1)
	double dKc_air;			//  Michaelis-Menten const of Rubisco for O2 at air temperature (Pa)
	double dKo_air;			//  Michaelis-Menten const of Rubisco for CO2 at air temperature (Pa)
	double dGanmaStar_air;	//   in the absence of mitochondria respiraton at air temperature (Pa)
	double dJm_air;			//  potential rate of electron transport rate per unit leaf area at air temperature (umol m-2 s-1)
	double dJ_air;			//  rate of electron transport rate per unit leaf area at air temperature (umol m-2 s-1)

	double dQstar;			// absorbed PAR (q*) when j leaf photosythesis is co-limited by Vcmax and Jmax (umol m-2 s-1) (Wang, 2000AFM)
	double dCs;				// CO2 concentration at leaf surface (ppm)

	// *******************************************************
	// Big-Leaf model
	// *******************************************************
	// convert degree C to K
	double dTemperature_K = dLeafTemperature + 273.15;
	double dAirTemperature_K = dAirTemperature + 273.15;
	
	// double dRow  = 1.293 / (1.0 + 0.00367 * dLeafTemperature );				// kg m-3
	double dRow = calculate_air_density(dLeafTemperature, dPressure, c_dSaturatedRH, 0);
	double dQ_air  = dVP_air
		/ (dPressure -  dVP_air);		// mixing ratio (kg kg-1)
	double dQ_leaf = dVP_leaf * 18.0 / 28.96 / (dPressure -  dVP_leaf);		// mixing ratio (kg kg-1)

	// convert relative humidity at leaf surface
	// rb * (Qsurf - Qair) = rs * (Qleaf - Qsurf)
	// Qsurf = (Gs * Qleaf + Gb * Qair) / (Gb + Gs)
	double dRhs;		// relative humidity at leaf surface (no dimension)
	double dQsurf = (dGc * dQ_leaf + dGb * dQ_air) / (dGb + dGc);			// mixing ratio (kg kg-1)
	double dVP_surf = dQsurf * dPressure / (18.0 / 28.96 + dQsurf);
	// dRhs = dVP_surf / dVP_leaf;
	dRhs = (dGb * dQ_air + dGc * dQ_leaf) / (dGb + dGc) / dQ_leaf;			// Su et al., 1996; Eq. 12

	// when Leuning model is applied, dRhs is updated to VPD 
	double dVPD_surf = (1.0 - dRhs) * dVP_surf;
	if(m_iLeuningModel) dRhs = 1.0 / (1.0 + dVPD_surf / m_dD0);

	if(dRhs <= 0.0) dRhs = 0.0;
	if(dRhs > 1.0) dRhs = 1.0;

	// calculate boundary layer conductance for CO2
	// dGb_co2 = 1.0 / pow((1.0 / dGb) * c_dRatio_Diff_ec, 2.0 / 3.0);

	// convert conductance unit from (m s-1) to (mol m-2 s-1)
	double dRow_mol  = dRow * 1000.0 / 28.96;	// air density (mol m-3) ; (28.96 g mol-1)
	const double dRatioGs = c_Ganmawc;			// mol ratio of CO2 to H2O 
	const double dRatioGb = 1.37;				// mol ratio of CO2 to H2O pow(1.6, 2/3)
	dGbc = dGb * dRow_mol / dRatioGb;			// (m s-1 mol m-3) -> (mol m-2 s-1)
	dGbw = dGb * dRow_mol;						// (m s-1 mol m-3) -> (mol m-2 s-1)
	
	// temperature responses basecd on Arrhenius Function
	if(m_iC3){
		// C3 photosynthesis
		temperature_response(dVCmax, dRd, dKc, dKo, dGanmaStar, dJm, dJ, dTemperature_K, dPPFD, dPressure, dTg, dUpscalingFactor);
		temperature_response(dVCmax_air
			, dRd_air, dKc_air, dKo_air, dGanmaStar_air, dJm_air, dJ_air, dAirTemperature_K, dPPFD, dPressure, dTg, dUpscalingFactor);

		//  calculate Effective Michaelis-Menten constant
		//  (Eq.3; de Pury and Farquhar, 1997)
		dKp = dKc * (1.0 + c_Oi / dKo);

		m_dJmax = dJm;
		m_dVCmax = dVCmax;

		// *******************************************************************************
		// Coupled photosynthesis and stomatal conductance model
		// *******************************************************************************
		double dCi_old = m_dErrorValue, dAn_old = m_dErrorValue;
		int iIteration = 0;
		const int iMaximum_iteration = 10;
		m_dCi = dCO2 * 0.7;						// initilization
		while(iIteration < iMaximum_iteration){
			// ***************************************************************************
			// Trying iteration
			// ***************************************************************************
			m_dAn = photosynthesis_model_C3(m_dCi, dVCmax, dRd, dKc, dKo, dGanmaStar, dJ, dPressure);
			dCs = dCO2 - m_dAn / dGbc;					// CO2 concentration at leaf surface (ppm)

			if(m_iLeuningModel == 0){
				m_dGsw = m_dBBm * m_dAn / dCs * dRhs + m_dBBb;
			}
			else{
				m_dGsw = m_dBBm * m_dAn / (dCs - dGanmaStar / (dPressure * 100.0) / pow(10, -6)) * dRhs + m_dBBb;
			}
			if(m_dGsw < m_dBBb) m_dGsw = m_dBBb;
			m_dGsc = m_dGsw / dRatioGs;	
			m_dCi = dCO2 - m_dAn * (m_dGsc + dGbc) / m_dGsc / dGbc;
			if( fabs(m_dCi - dCi_old) < 1.0 || fabs(m_dAn - dCi_old) < 0.1 ){
				break;
			}
			dCi_old = m_dCi;
			dAn_old = m_dAn;
			iIteration++;
		}

		if(iIteration == iMaximum_iteration){			
			// ***************************************************************************
			// In case of instability for the iterration
			// ***************************************************************************
			double dEa_vcmax = 1.0;				// Michaelis-Menten const (Eq.2; de Pury and Farquhar, 1997)
			double dEa_j = 4.0;					// Michaelis-Menten const (Eq.4; de Pury and Farquhar, 1997)
			double dMc_j = 8.0 * dGanmaStar;	// Michaelis-Menten const (Pa) (Eq.4; de Pury and Farquhar, 1997)
			CubicEquation(dAv, dVCmax, dEa_vcmax, dKp   , dCO2, dGanmaStar, dRhs, dRd, dGbw, dPressure);
			CubicEquation(dAj, dJ    , dEa_j    , dMc_j , dCO2, dGanmaStar, dRhs, dRd, dGbw, dPressure);
			m_dAv = dAv;
			m_dAj = dAj;

			if(dAv == m_dErrorValue || dAj == m_dErrorValue){
				m_dAn = m_dErrorValue;
				m_dTranspiration = m_dErrorValue;
				m_dGsc = m_dErrorValue;
				m_dGsw = m_dErrorValue;
				m_dCi = m_dErrorValue;
				m_dQstar = m_dErrorValue;
				return;
			}

			// co-limitation between Av and Aj 
			// Collatz et al., 1991 : Agricultural and Foreest Meteorology, 54, 107-136.
			// Physiological and environmental regulation of stomatal conductance, photosynthesis and transpiration: a model that includes a laminar boundary layer
			m_dAn = ( (dAv + dAj) - sqrt( pow( (dAv + dAj), 2) - 4.0 * c_dS * dAv * dAj) ) / 2.0 / c_dS;
			if(m_dAn < - dRd){
				if(dAv < 0.0){
					m_dAn = dAj;
				}
			}
		}
	}
	else{
		// C4 photosynthesis
		photosynthesis_model_C4(m_dAn, dCO2, dPPFD,  dLeafTemperature, dRhs, dPressure, dUpscalingFactor);
	}

	// ***************************************************
	// calculating gs using Ball-Berry model
	// ***************************************************
	dCs = dCO2 - m_dAn / dGbc;							// CO2 concentration at leaf surface (ppm)
	m_dGsw = m_dBBm * m_dAn / dCs * dRhs + m_dBBb;		// stomatal conductance of H2O (mol m-2 s-1)
	if(m_iLeuningModel){
		m_dGsw = m_dBBm * m_dAn / (dCs - dGanmaStar / (dPressure * 100.0) * pow(10, 6)) * dRhs + m_dBBb;
	}
	if(m_dGsw < m_dBBb) m_dGsw = m_dBBb;
	m_dGsc = m_dGsw / dRatioGs;							// stomatal conductance of CO2 (mol m-2 s-1)
		
	if(m_dGsc != 0.0 && dGbc != 0.0){
		m_dCi = dCO2 - m_dAn * (m_dGsc + dGbc) / m_dGsc / dGbc;
	}else{
		m_dCi = dCO2;
	}

	// ***************************************************
	// added respiration, because optimization was done for gross photosynthesis rather than net photosynthesis
	// ***************************************************
	// m_dAn = m_dAn + dRd;
	double dRd_zero = 0.0;
	if(m_iC3){
		m_dAn = photosynthesis_model_C3(m_dCi, dVCmax, dRd_zero, dKc, dKo, dGanmaStar, dJ, dPressure);
		dAv = m_dAv;
		dAj = m_dAj;
	}
	else{
		double dAlfa_collatz = 0.8;		// Leaf quantum absorbance (Table 1; Collatz et al., 1992)
		double dAlfaR_collatz = 0.11;	// RuBP quantum requirement (mol mol-1) (Table 1; Collatz et al., 1992)
		double dAlfaP_collatz = 0.167;	// PEP quantum requirement (mol mol-1) (Table 1; Collatz et al., 1992)
		double dF_collatz = 0.6;		// Fractional RuBP quantum requirement (Table 1; Collatz et al., 1992)
		double dKp_collatz = 0.7;		// PEPcase rate constant for CO2 (mol m-2 s-1) (Table 1; Collatz et al., 1992)

		// calculating Vcmax at given temperature (Eq. 5B)
		double dQ10_c4 = 2.0;
		m_dVCmax = m_Vcmax25 * pow(dQ10_c4, (dLeafTemperature - 25.0) / 10.0);
		m_dVCmax /= (1.0 + exp(0.3 * (13.0 - dLeafTemperature) ) );
		m_dVCmax /= (1.0 + exp(0.3 * (dLeafTemperature - 55.0) ) );

		dAv = (dCO2 * pow(10, -6)) * dKp_collatz * pow(10, 6);
		dAj = dAlfa_collatz * dAlfaR_collatz * dF_collatz * dPPFD;

		m_dJmax = m_dErrorValue;
	}

	// ***************************************************
	// calculating effect of stomatal and boundary layer resistance into photosynthesis
	// Farquhar and Sharkey, 1982 : Stomatal conductance and photosynthesis, Annu. Rev. Physiol. 33, 317-345.
	// ***************************************************
	if(m_iC3){
		m_dA_gs = photosynthesis_model_C3(dCs, dVCmax, dRd_zero, dKc, dKo, dGanmaStar, dJ, dPressure);			// gross rather than net
		m_dA_gs_ga = photosynthesis_model_C3(dCO2, dVCmax, dRd_zero, dKc, dKo, dGanmaStar, dJ, dPressure);		// gross rather than net
		m_dA_gs_ga_ts = photosynthesis_model_C3(dCO2, dVCmax_air, dRd_zero, dKc_air, dKo_air, dGanmaStar_air, dJ_air, dPressure);	// gross rather than net
	}
	else{
		photosynthesis_model_C4(m_dA_gs, dCs, dPPFD,  dLeafTemperature, dRhs, dPressure, 0.0);
		photosynthesis_model_C4(m_dA_gs_ga, dCO2, dPPFD,  dLeafTemperature, dRhs, dPressure, 0.0);
		photosynthesis_model_C4(m_dA_gs_ga_ts, dCO2, dPPFD,  dAirTemperature, dRhs, dPressure, 0.0);
	}

	// ***************************************************
	// calculate transpiration 
	// ***************************************************
	double dGtw = 1.0 / (1.0 / m_dGsw + 1.0 / dGbw ) / dRow_mol;			// (m s-1)
	m_dTranspiration = dRow * dGtw * (dQ_leaf - dQ_air ) * pow(10, 6);		// mg m-2 s-1

	// ***************************************************
	// daytime respiration and non-limited photosynthesis
	// ***************************************************
	m_dRd = dRd;
	m_dAv = dAv;
	m_dAj = dAj;

	// ***************************************************
	// absorbed PAR (q*) when j leaf photosythesis is co-limited by Vcmax and Jmax (umol m-2 s-1) (Eq. 5 in Wang, 2000AFM)
	// ***************************************************
	double dB = m_dJmax / m_dVCmax;
	double cC_p = (dPressure * 100.0) * (m_dCi * pow(10, -6));								// interceller CO2 pressure (Pa)
	double dBeta = 4.0 * ( cC_p + 2.0 * dGanmaStar) / (cC_p + dKp );
	dQstar = dBeta * (c_sl * dBeta - dB) / ((1.0 - c_f) / 2.0) / (dBeta - dB) * m_dVCmax;

	// absorbed PAR (q*) when j leaf photosythesis is co-limited by Vcmax and Jmax (umol m-2 s-1) (Eq. 25 in de Pury & Farquhar, 1997)
	dQstar = m_dVCmax * 2.0 / (1.0 - c_f);
	dQstar *= 4.0 * (cC_p + 2.0 * dGanmaStar) * (4.0 * c_sl * (cC_p + 2.0 * dGanmaStar) - m_dJmax / m_dVCmax * (cC_p + dKp) );
	dQstar /= ((cC_p + dKp) * (4.0 * (cC_p + 2.0 * dGanmaStar) - m_dJmax / m_dVCmax * (cC_p + dKp) ));

	if(dQstar < 0.0) dQstar = 0.0;
	m_dQstar = dQstar;
	if(m_iC3 != 0) m_dQstar = m_dErrorValue;

	// ***************************************************
	// if Vcmax25 greater than Jmax25, error value is set. (penalty)
	// ***************************************************
	if(m_dCi < 0.0 || m_dCi / dCO2 > 1.5){
//		m_dAn = m_dErrorValue;
		m_dTranspiration = m_dErrorValue;
		m_dGsc = m_dErrorValue;
		m_dGsw = m_dErrorValue;
		m_dCi = m_dErrorValue;
		m_dQstar = m_dErrorValue;
	}
}

void CLeafPhotosynthesis::Stomatal_conductance_Ball_Berry(const double& dCO2, const double& dPPFD, const double& dGPP, double dLeafTemperature, 
	const double& dVPD_air, const double& dVP_air, const double& dVP_leaf, const double& dPressure, const double& dGc, double dGb, const double& dTg, const double& dUpscalingFactor)
{	
	double dQ_air  = dVP_air * 18.0 / 28.96 / (dPressure -  dVP_air);		// mixing ratio (kg kg-1)
	double dQ_leaf = dVP_leaf * 18.0 / 28.96 / (dPressure -  dVP_leaf);		// mixing ratio (kg kg-1)

	// relative humidity at leaf surface (no dimension)
	double dRhs = (dGb * dQ_air + dGc * dQ_leaf) / (dGb + dGc) / dQ_leaf;			// Su et al., 1996; Eq. 12

	// convert relative humidity at leaf surface
	// rb * (Qsurf - Qair) = rs * (Qleaf - Qsurf)
	// Qsurf = (Gs * Qleaf + Gb * Qair) / (Gb + Gs)
	double dQsurf = (dGc * dQ_leaf + dGb * dQ_air) / (dGb + dGc);			// mixing ratio (kg kg-1)
	double dVP_surf = dQsurf * dPressure / (18.0 / 28.96 + dQsurf);

	// when Leuning model is applied, dRhs is updated to VPD 
	double dVPD_surf = (1.0 - dRhs) * dVP_surf;
	if(m_iLeuningModel) dRhs = 1.0 / (1.0 + dVPD_surf / m_dD0);

	if(dRhs <= 0.0) dRhs = 0.0;
	if(dRhs > 1.0) dRhs = 1.0;

	// convert conductance unit from (m s-1) to (mol m-2 s-1)
	// double dRow  = 1.293 / (1.0 + 0.00367 * dLeafTemperature );					// kg m-3
	double dRow = calculate_air_density(dLeafTemperature, dPressure, c_dSaturatedRH, 0);
	double dRow_mol  = dRow * 1000.0 / 28.96;			// air density (mol m-3) ; (28.96 g mol-1)
	const double dRatioGs = c_Ganmawc;					// mol ratio of CO2 to H2O 
	const double dRatioGb = 1.37;						// mol ratio of CO2 to H2O pow(1.6, 2/3)
	double dGbc = dGb * dRow_mol / dRatioGb;			// (m s-1 mol m-3) -> (mol m-2 s-1)
	double dGbw = dGb * dRow_mol;						// (m s-1 mol m-3) -> (mol m-2 s-1)

	// ***************************************************
	//  test for applicability for photosynthetic model
	// ***************************************************
	double dJm;			//  potential rate of electron transport rate per unit leaf area (umol m-2 s-1)
	double dKo;			//  Michaelis-Menten const of Rubisco for O2 (Pa)
	double dKc;			//  Michaelis-Menten const of Rubisco for CO2 (Pa)

	double dAv;			//  Rubisco-limited photosynthesis (umol m-2 s-1)
	double dAj;			//  Electron-transport limited rate of photosynthesis (umol m-2 s-1)
	double dAn;			//  net photosynthesis rate (umol m-2 s-1)
	double dRd;			// leaf respiration per unit leaf area (umol m-2 s-1)
	double dJ;			//  rate of electron transport rate per unit leaf area (umol m-2 s-1)
	double dVCmax;		//  photosynthesis Rubisco canopy per unit leaf area (umol m-2 s-1)
	double dKp;			//  Effective Michaelis-Menten constant of Rubisco (Pa)
	double dGanmaStar;	//   in the absence of mitochondria respiraton (Pa)
	double dTemperature_K = dLeafTemperature + 273.15;

	if(m_iC3){
		// C3 photosynthesis 

		// temperature responses basecd on Arrhenius Function
		temperature_response(dVCmax, dRd, dKc, dKo, dGanmaStar, dJm, dJ, dTemperature_K, dPPFD, dPressure, dTg, dUpscalingFactor);

		//  calculate Effective Michaelis-Menten constant
		//  (Eq.3; de Pury and Farquhar, 1997)
		dKp = dKc * (1.0 + c_Oi / dKo);

		m_dJmax = dJm;
		m_dVCmax = dVCmax;

		double dEa_vcmax = 1.0;				// Michaelis-Menten const (Eq.2; de Pury and Farquhar, 1997)
		double dEa_j = 4.0;					// Michaelis-Menten const (Eq.4; de Pury and Farquhar, 1997)
		double dMc_j = 8.0 * dGanmaStar;	// Michaelis-Menten const (Eq.4; de Pury and Farquhar, 1997)
		CubicEquation(dAv, dVCmax, dEa_vcmax, dKp   , dCO2, dGanmaStar, dRhs, dRd, dGbw, dPressure);
		CubicEquation(dAj, dJ    , dEa_j    , dMc_j , dCO2, dGanmaStar, dRhs, dRd, dGbw, dPressure);
		dAn = ( (dAv + dAj) - sqrt( pow( (dAv + dAj), 2) - 4.0 * c_dS * dAv * dAj) ) / 2.0 / c_dS - dRd;
	}
	else{
		// C4 photosynthesis 
		photosynthesis_model_C4(dAn, dCO2, dPPFD,  dLeafTemperature, dRhs, dPressure, dUpscalingFactor);
		dAv = dAn;
		dAj = dAn;
	}

	// ***************************************************
	// calculating gs using Ball-Berry model
	// ***************************************************
	double dCs = dCO2 - dGPP / dGbc;					// CO2 concentration at leaf surface (ppm)
	m_dGsw = m_dBBm * dGPP / dCs * dRhs + m_dBBb;		// stomatal conductance of H2O (mol m-2 s-1)
	if(m_iLeuningModel){
		double dGanmaStar;
		double dTemperature_K = dLeafTemperature + 273.15;
		GanmaStar(dGanmaStar, dTemperature_K, dPressure);
		m_dGsw = m_dBBm * dGPP / (dCs - dGanmaStar / (dPressure * 100.0) * pow(10, 6)) * dRhs + m_dBBb;
	}
	if(m_dGsw < m_dBBb) m_dGsw = m_dBBb;
	m_dGsc = m_dGsw / dRatioGs;							// stomatal conductance of CO2 (mol m-2 s-1)
		
	if(m_dGsc != 0.0 && dGbc != 0.0){
		m_dCi = dCO2 - dGPP * (m_dGsc + dGbc) / m_dGsc / dGbc;
	}else{
		m_dCi = dCO2;
	}

	// calculate transpiration 
	double dGtw = 1.0 / (1.0 / m_dGsw + 1.0 / dGbw ) / dRow_mol;			// (m s-1)
	m_dTranspiration = dRow * dGtw * (dQ_leaf - dQ_air ) * pow(10, 6);		// mg m-2 s-1

	// error value is set. (penalty)
	if(m_dCi < 0.0 || ( dGPP > 0.0 && m_dCi / dCO2 > 1.0) || dAv < -10 || dAj < -10 ){
		m_dAn = m_dErrorValue;
		m_dTranspiration = m_dErrorValue / 10.0;
	}
}

void CLeafPhotosynthesis::ExtendedBigLeaf_Vcmax(const double& dCO2, double dLeafTemperature, const double& dPressure, const double& dPPFD, 
	const double& dGPP, const double& dET, const double& dGc, double dGb, const double& dSaturatedLightLevel, double dRd_eBL)
{
	// calculating GPP by using Kosugi et al., 2005, 2013
	// leaf photosynthesis by Farquhar Model based on de Pury and Farquhar (1997)
	// dCO2					: atmospheric CO2 concentration (ppm)
	// dLeafTemperature		: leaf temperature (degree C)
	// dPressure			: atmospheric pressure (hPa)
	// dGPP					: gross primary productivity (umol m-2 s-1)
	// dET					: water vapor flux (mol m-2 s-1)
	// dGc					: surface layer conductance condutance (m s-1)
	// dGb					: boudary layer conductance condutance of H2O (m s-1)

	if(dPPFD < dSaturatedLightLevel || dCO2 == m_dErrorValue || dLeafTemperature == m_dErrorValue || dPressure == m_dErrorValue || 
		dPPFD == m_dErrorValue || dGPP == m_dErrorValue || dET == m_dErrorValue || dGc == m_dErrorValue || dGb == m_dErrorValue){
		m_dVCmax_eBL = m_dErrorValue;
		m_dVCmax25_eBL = m_dErrorValue;
		m_dCi_eBL = m_dErrorValue;
		return;
	}	
	
	// convert degree C to K
	double dTemperature_K = dLeafTemperature + 273.15;
	
	// calculating total conductance for CO2 (mol m-2 s-1) (Eq.9; Kosugi et al., 2005)
	// double dRow_mol  = 1.293 / (1.0 + 0.00367 * dLeafTemperature ) * 1000.0 / 28.96;	// air density (mol m-3) ; (28.96 g mol-1)
	double dRow_mol = calculate_air_density(dLeafTemperature, dPressure, c_dSaturatedRH, 1);
	const double dRatioGs = c_Ganmawc;			// mol ratio of CO2 to H2O 
	const double dRatioGb = 1.37;				// mol ratio of CO2 to H2O pow(1.6, 2/3)
	double dGtc;								// total conductance for CO2 (mol CO2 m-2 s-1)
	dGtc = 1.0 / (1.0 / dRow_mol / (dGb / dRatioGb) + 1.0 / dRow_mol / (dGc / dRatioGs) );

	// calculating partial pressure of CO2 at the site of carboylation (Pa) (Eq.8; Kosugi et al., 2005)
	double dCc_p;
	dCc_p = ( (dGtc - dET / 2.0) * dCO2 - (dGPP - dRd_eBL) ) / (dGtc + dET / 2.0) * (dPressure / pow(10, 4));

	// calculating partial pressure of interceller CO2 (Pa) (Eq.8; Kosugi et al., 2005)
	double dCi;
	dCi = dCc_p / (dPressure / pow(10, 4));
	m_dCi_eBL = dCi;		// ppm

	// calculating CO2 compensation point without non-photorespiratory respiration (Pa)
	//  (Eq.9; de Pury and Farquhar, 1997)
	double dGammaStar;
	dGammaStar = 3.69 + 0.188 * (dTemperature_K - 298.15) + 0.0036 * pow((dTemperature_K - 298.15), 2);

	// calculating RuBP-saturated rate of carboyxlation (Wc; umol m-2 s-1) (Eq.6; Kosugi et al., 2005)
	double dKo;			//  Michaelis-Menten const of Rubisco for O2 (Pa)
	double dKc;			//  Michaelis-Menten const of Rubisco for CO2 (Pa)
	dKc    = Arrhenius(c_kc, c_dEa_Kc, dTemperature_K);
	dKo    = Arrhenius(c_ko, c_dEa_Ko, dTemperature_K);

	// calculating photosynthesis (Eq.4; Kosugi et al., 2005)
	m_dVCmax_eBL = dGPP / (dCc_p / (dCc_p + dKc * (1.0 + c_Oi / dKo) ) * (1.0 - dGammaStar / dCc_p) );

	// calculating leaf respiration
	double dVcmax_25;			// leaf respiration per unit leaf area at 25 oC (umol m-2 s-1)
	dVcmax_25 = Inverse_Arrhenius(m_dVCmax_eBL, c_dEa_Vl, dTemperature_K);
	m_dRd_eBL = (Arrhenius(0.0089 * dVcmax_25, c_dEa_Rl, dTemperature_K) + dRd_eBL ) / 2.0;	// (Eq.9; de Pury and Farquhar, 1997)
	m_dVCmax25_eBL = dVcmax_25;

	if(fabs(m_dRd_eBL - dRd_eBL ) / m_dRd_eBL < 0.01 || m_iIteration_eBL > 5000){
		return;
	}
	m_iIteration_eBL++;

	// iterration using updated leaf respiration
	ExtendedBigLeaf_Vcmax(dCO2, dLeafTemperature, dPressure, dPPFD, dGPP, dET, dGc, dGb, dSaturatedLightLevel, m_dRd_eBL);
}

double CLeafPhotosynthesis::Arrhenius(const double& k25, const double& Ea, const double& dT)
{	
	//Compute Arrhenius Form for Farquhar Model
	return k25 * exp(Ea * (dT - 298.15) / (298.15 * c_R * dT));
}

double CLeafPhotosynthesis::Arrhenius2(const double& k25, const double& dC, const double& dHa, const double& dT)
{	
	//Compute Arrhenius Form for Bernacchi et al. 2001
	// dT (K)
	return k25 * exp(dC - dHa * 1000.0 / c_R / dT);
}

double CLeafPhotosynthesis::Arrhenius3(const double& k25, const double& dQ10, const double& dT)
{	
	//Compute Arrhenius Form for Bernacchi et al. 2001
	// dT (K)
	return k25 * pow(dQ10, (dT - 273.15 - 25.0) / 10.0 );
}

double CLeafPhotosynthesis::Arrhenius4(const double& k25, const double& dHa, const double& dT)
{	
	//Compute Arrhenius Form for Bernacchi et al. 2001
	// dT (K)
	return k25 * exp( (1.0 - 298.15 / dT) * dHa / c_R / 298.15 );
}

double CLeafPhotosynthesis::Inverse_Arrhenius(const double& k, const double& Ea, const double& dT)
{	
	//Compute Arrhenius Form for Farquhar Model
	return k / exp(Ea * (dT - 298.15) / (298.15 * c_R * dT));
}

void CLeafPhotosynthesis::CubicEquation(double& _An, double& Aa, double& Ea, double& Da, const double& dCO2, 
					double& dGanmaStar, const double& dRhs, const double& dRd, const double& dGb, const double& dPressure)
{
	// Su et al., 1997: Development of a coupled leaf and canopy model for the simulation of Plant-Atmosphere interaction
	// Journal of Applied Meteorology, 35, 733-748.

	// _An			: photosynthetic rate (umol m-2 s-1)
	// Aa			: Vcmax or Jmax (umol m-2 s-1)
	// Ea			: activation energy
	// Da			: Michaelis-Menten constant (Pa)
	// dCO2			: atmospheric CO2 concentration (ppm)
	// dGanmaStar	: * in the absence of mitochondria respiraton (Pa)
	// dRhs			: relative humidity at leaf surface (--)
	// dRd			: dark respiration (umol m-2 s-1)
	// dGb			: boundary layer condctance for H2O (mol m-2 s-1)
	// dPressure	: atmospheric pressure (hPa)

	double Ra, Sita;
	double Ca0, Ca1, Ca2, Ca3;
	double GanmaA;
	double AlfaA, BetaA, SitaA;
	double GanmaS_ppm, Da_ppm;

	// Pa  ppm
	GanmaS_ppm = dGanmaStar / (dPressure * pow(10,2)) * pow(10,6);
	Da_ppm     = Da / (dPressure * pow(10,2)) * pow(10,6);

	if(dGb != 0.0){
		AlfaA = c_Ganmawc + m_dBBb / dGb - m_dBBm * dRhs;				// None Dimension 
	}
	else{
		AlfaA = 0.0;
	}
	BetaA = dCO2 * (dGb * (m_dBBm * dRhs - c_Ganmawc) - 2.0 * m_dBBb);	// ppm mol m-2 s-1
	SitaA = dGb * m_dBBm * dRhs - m_dBBb;								// mol m-2 s-1
	GanmaA = pow(dCO2, 2) * m_dBBb * dGb;								// ppm2 (mol m-2 s-1)^2

	// Cubic Eauation Parameter
	Ca0 = GanmaA * (Aa * (GanmaS_ppm / dCO2 - 1.0) 
		+ 	dRd * (Ea + Da_ppm / dCO2));					// (ppm mol m-2 s-1)^3
	Ca0 *= pow(10, -18);									// (mol m-2 s-1)^3
	Ca1 = (Ea + Da_ppm / dCO2) * GanmaA + 
		dRd * (Ea * BetaA + Da_ppm * SitaA) + 
		Aa * (GanmaS_ppm * SitaA - BetaA);					// (ppm mol m-2 s-1)^2
	Ca1 *= pow(10, -12);									// (mol m-2 s-1)^2

	Ca2 = Ea * BetaA + Da_ppm * SitaA + 
		(Ea * dRd - Aa) * AlfaA;							// ppm mol m-2 s-1
	Ca2 *= pow(10, -6);										// (mol m-2 s-1)

	Ca3 = Ea * AlfaA;										// Nondimensional

	Ra = pow(pow(Ca2 / (3.0 * Ca3), 2) - 
		Ca1 / (3.0 * Ca3), 3.0 / 2.0 );						// (mol m-2 s-1)^3

	if(fabs(-(Ca0 / Ca3	- Ca1 * Ca2 / (3.0 * pow(Ca3, 2)) + 2.0 * pow(Ca2, 3)/(27.0 * pow(Ca3, 3))) / (2.0 * Ra)) <= 1.0){
		Sita = acos(-(Ca0 / Ca3	- Ca1 * Ca2 / (3.0 * pow(Ca3, 2))
			+ 2.0 * pow(Ca2, 3)/(27.0 * pow(Ca3, 3))) / (2.0 * Ra)) / 3.0;
	}
	else{
		_An = m_dErrorValue;
		return;
	}

	// Compute photosynthesis (mol m-2 s-1)
	_An = 2.0 * pow(Ra, 1.0 / 3.0) * cos(Sita + 4.0 / 3.0 * 3.14) - Ca2 / (3.0 * Ca3);
	_An *= pow(10, 6);	
}

void CLeafPhotosynthesis::CubicEquation_Baldocchi(double& _An, double& Aa, double& Ea, double& Da, const double& dCO2, 
					double& dGanmaStar, const double& dRhs, const double& dRd, const double& dGb_q, const double& dPressure)
{
	// Baldocchi et al., 1994: An analytical solution for coupled leaf photosynthesis and stomatal conductance models
	// Tree Physiology, 34, 1069-1079.

	// _An			: photosynthetic rate (umol m-2 s-1)
	// Aa			: Vcmax or Jmax (umol m-2 s-1)
	// Ea			: 
	// Da			: Michaelis-Menten constant (Pa)
	// dCO2			: atmospheric CO2 concentration (ppm)
	// dGanmaStar	: * in the absence of mitochondria respiraton (Pa)
	// dRhs			: relative humidity at leaf surface (--)
	// dRd			: dark respiration (umol m-2 s-1)
	// dGb			: boundary layer condctance for H2O (mol m-2 s-1)
	// dPressure	: atmospheric pressure (hPa)

	// Pa  ppm
	double GanmaS_ppm = dGanmaStar / (dPressure * pow(10,2)) * pow(10,6);
	double Da_ppm     = Da / (dPressure * pow(10,2)) * pow(10,6);

	// conductance parameters from H2O to CO2
	const double dRatioGb = 1.37;				// mol ratio of CO2 to H2O pow(1.6, 2/3)
	double dGb_c = dGb_q / dRatioGb;
	double dBBb_c = m_dBBb / c_dRatio_Diff_ec;
	double dBBm_c = m_dBBm / c_dRatio_Diff_ec;

	const double dPI = 3.15;
	double a = Aa;				// (umol m-2 s-1)
	double b = Da_ppm;			// (ppm)
	double d = GanmaS_ppm;		// (ppm)
	double e = Ea;				// Nondimensional	

	// Equ. 11
	double dAlfaA = 1.0 + dBBb_c / dGb_c - dBBm_c * dRhs;					// Nondimensional

	// Eq. 12
	double dBetaA = dCO2 * (dGb_c * dBBm_c * dRhs - 2.0 * dBBb_c - dGb_c);	// ppm mol m-2 s-1

	// Eq. 13
	double dGanmaA = pow(dCO2, 2) * dBBb_c * dGb_c;							// ppm2 (mol m-2 s-1)^2

	// Eq. 14
	double dSitaA = dGb_c * dBBm_c * dRhs - dBBb_c;							// mol m-2 s-1 

	// Eq. 8
	double p = (e * dBetaA + b * dSitaA - a * dAlfaA + e * dAlfaA * dRd) / (e * dAlfaA);	// ppm mol m-2 s-1
	double q = (e * dGanmaA + b * dGanmaA / dCO2 - a * dBetaA + a * d * dSitaA + e * dRd * dBetaA + dRd * b * dSitaA) / (e * dAlfaA);	// (ppm mol m-2 s-1)^2
	double r =  (- a * dGanmaA + a * d * dGanmaA / dCO2 + e * dRd * dGanmaA + dRd * b * dGanmaA / dCO2) / (e * dAlfaA);	// (ppm mol m-2 s-1)^3

	// solution of cubic equation (Eq. 15 - 17)
	// Eq.	16
	double dQ = ( p * p - 3.0 * q ) / 9.0;								// (umol m-2 s-1)^2
	double dR = ( 2.0 * p * p * p - 9.0 * p * q + 27.0 * r ) / 54.0;	// (umol m-2 s-1)^3

	// Eq. 18
	double dSita = acos( dR / sqrt(dQ * dQ * dQ) );						// Nondimensional

	// Eq. 16
	double dX3 = -2.0 * sqrt(dQ) * cos( (dSita + 4.0 * dPI) / 3.0 ) - p / 3.0;	// ppm mol m-2 s-1
	_An = dX3 * 28.96 / 44.01;	//(umol_co2 mol_air-1) * (mol_co2 m-2 s-1) * (mol_air) / (mol_co2)
}

void CLeafPhotosynthesis::photosynthesis_model_C4(double& dAn, const double& dCO2, const double& dPPFD, const double& dT_leaf, const double& dRhs, const double& dPressure, const double& dUpscalingFactor)
{
	// photosynthesis model for C4 vegetation
	// Collatz, G, Ribas-Carbo, M., Berry J. A. (1992) Aust. J. Plant Physiol, 19, 519-538.
	//   Coupled photosynthesis-stomatal conductance model for leaves of C4 plants

	// dCO2			: atmospheric CO2 concentration (ppm)
	// dT_leaf      : leaf temperature (degree C)
	// dRhs			: relative humidity at leaf surface (--)
	// dRd			: dark respiration (umol m-2 s-1)
	// dPressure		: atmospheric pressure (hPa)
	// m_dBBb       :  b by Ball Berry model (mol m-2 s-1)

	double dA, dB, dC, dD;
	double dQ, dR, dS;
	double dCO2p;							// partial pressure of CO2 (Pa Pa-1)
	const double c_dBeta = 0.93;			// Curvature parameter (--) (Table 2 in Collatz et al., 1992)

	double dKt;								// initial slople of photosynthetic CO2 response  (Table 2 & Eq. 5B)
	double dM;								// flux determined by the rubisco and light limited capacities (mol m-2 s-1) (Eq. 2B)

	// calculating CO2 pressure (Pa) from mixing ratio
	dCO2p = (dPressure * 100.0) * (dCO2 * pow(10, -6));	// interceller CO2 pressure (Pa)
	double dPressureRatio = (dPressure * 100.0) / dCO2p;

	const double c_dK = 0.7;				// initial slople of photosynthetic CO2 response (mol m-2 s-1) at 25 degree C (Table 2)
	const double c_dQ10 = 2.0;				// temperature sensitivity factor (Table 2)
	dKt = c_dK * pow(c_dQ10, (dT_leaf - 25.0) / 10.0);

	const double c_dRd = 0.8;				// leaf respiration (umol m-2 s-1) at 25 degree C (Table 2)
	double dRd;								// leaf respiration (mol m-2 s-1)
	dRd = c_dRd *  pow(c_dQ10, (dT_leaf - 25.0) / 10.0) / (1.0 + exp(1.3 * (dT_leaf - 55.0) ) );
	dRd /= pow(10, 6);						// umol m-2 s-1 to mol m-2 s-1
	dRd *= dUpscalingFactor;				// upscaling from leaf to canopy scale

	// calculating Vcmax at given temperature (Eq. 5B)
	double dVt;								// Vcmax (mol m-2 s-1)
	dVt = m_Vcmax25 * pow(c_dQ10, (dT_leaf - 25.0) / 10.0);
	dVt /= (1.0 + exp(0.3 * (13.0 - dT_leaf) ) );
	dVt /= (1.0 + exp(0.3 * (dT_leaf - 55.0) ) );
	dVt /= pow(10, 6);						// umol m-2 s-1 to mol m-2 s-1

	// calculating photosynthesis determined by the rubisco and light limited capacities (mol m-2 s-1) (Eq. 2B)
	double dQE_a, dQE_b, dQE_c;
	dQE_a = 0.83;							// si-ta in Table 2
	dQE_b = ( dVt *  pow(10, 6) ) + 0.8 * 0.067 * dPPFD;		// (umol m-2 s-1)
	dQE_c = ( dVt *  pow(10, 6) ) * 0.8 * 0.067 * dPPFD;		// (umol m-2 s-1)^2
	dM = ( dQE_b - sqrt( dQE_b * dQE_b - 4.0 * dQE_a * dQE_c) ) / 2.0 / dQE_a;
	dM /= pow(10, 6);						// (umol m-2 s-1) to (mol m-2 s-1)

	// Cubic Equation (Eq. 1C in Collatz et al., 1992) 
	dA = c_dBeta * m_dBBm * dRhs * dPressureRatio;									// (--)
	dB = c_dBeta * m_dBBb - c_dBeta * m_dBBm * dRhs * dRd * dPressureRatio;			// (mol m-2 s-1)
	dB += c_dRatio_Diff_ec * dKt - dM * m_dBBm * dRhs * dPressureRatio - dKt * m_dBBm * dRhs;
	dC = dRd * dM * m_dBBm *  dRhs * dPressureRatio - dM * m_dBBb + dRd * dKt * m_dBBm * dRhs - dKt * m_dBBb / dPressureRatio;
	dC += - c_dRatio_Diff_ec * dKt * dRd + dM * dKt * m_dBBm * dRhs - c_dRatio_Diff_ec * dKt * dM;						// (mol m-2 s-1)^2
	dD = dM * dKt * m_dBBb / dPressureRatio + c_dRatio_Diff_ec * dM * dKt * dRd - dM * dKt * m_dBBm * dRhs * dRd;		// (mol m-2 s-1)^3

	// Cubic Equation (Eq. 2C in Collatz et al., 1992) 
	dQ = ( ( dB / dA ) * ( dB / dA ) - 3.0 * dC / dA ) / 9.0;															// (mol m-2 s-1)^2
	dR = ( 2.0 * ( dB / dA ) * ( dB / dA ) * ( dB / dA ) - 9.0 * dB * dC / dA / dA + 27.0 * dD / dA ) / 54.0;			// (mol m-2 s-1)^3
	dS = acos( dR / sqrt(dQ * dQ * dQ));
	dAn = -2.0 * sqrt(dQ) * cos( (dS + 4.0 * 3.14) / 3.0 ) - dB / 3.0 / dA;												// (mol m-2 s-1)
	dAn *= pow(10, 6);		
	if(fabs(dR / sqrt(dQ * dQ * dQ)) > 1.0){
		dAn = m_dErrorValue /100.0;
	}																					// (umol m-2 s-1)

	// calculating photosynthesis determined by the rubisco and light limited capacities (mol m-2 s-1) (Eq. 3B)
//	dQE_a = c_dBeta;							// beta in Table 2
//	dQE_b = dM + dKt * dCO2p;					// (mol m-2 s-1)
//	dQE_c = dM * dKt * dCO2p;					// (mol m-2 s-1)
//	dAn = ( dQE_b - sqrt( dQE_b * dQE_b - 4.0 * dQE_a * dQE_c) ) / 2.0 / dQE_a;		// (mol m-2 s-1)
//
//	dAn = dAn * pow(10, 6);					// (mol m-2 s-1) to (umol m-2 s-1)
}

void CLeafPhotosynthesis::temperature_response(double& dVCmax, double& dRd, double& dKc, double&  dKo, double& dGanmaStar, double& dJm, double& dJ, double& dTemperature_K, const double& dPPFD, const double& dPressure, const double& dTg, const double& dUpscalingFactor)
{
	// dVCmax			: photosynthetic Rubisco capacity per unit leaf area (mol m-2 s-1)
	// dRd				: dark respiration rate per unit leaf area (mol m-2 s-1)
	// dKc				: Michaelis-Menten constant of Rubisco for CO2 (Pa)
	// dKo				: Michaelis-Menten constant of Rubisco for O2 (Pa)
	// dGanmaStar		: CO2 compensation point in absence of mitochondrial respiration (Pa)
	// dJm				: potential rate of electron transport rate per unit leaf area (mol m-2 s-1)
	// dJ				: rate of electron transport rate per unit leaf area (mol m-2 s-1)
	// dTemperature_K	: leaf temperature (K)
	// dPPFD			: photosynthetically photon flux density (mol m-2 s-1)

	double dIle;		//  PAR effectively absorbed by PSU (mol m-2 s-1)

	if(m_iPhotosynthesisFunctions == 0){
		// de Pury & Farquhar, 1997: Simple scaling of photosynthesis from leaves to canopies without the errors of big-leaf models
		//            Plant, Cell and Environment, 20, 537-557.
		// constant
		double dEa_Vl = 64800.0;
		double dEa_Rl = 66400.0;
		double dEa_Kc = 59400.0;
		double dEa_Ko = 36000.0;
		double dS = 710.0;			// electron-transport temperature response parameter (J K-1 mol-1)
		double dH = 220000.0;		// curvature parameter of Jm (J mol-1)

		// temperature responses basecd on Arrhenius Function
		dVCmax = Arrhenius(m_Vcmax25, dEa_Vl, dTemperature_K);
		dRd    = Arrhenius(0.0089 * m_Vcmax25, dEa_Rl, dTemperature_K);
		dKc    = Arrhenius(c_kc, dEa_Kc, dTemperature_K);
		dKo    = Arrhenius(c_ko, dEa_Ko, dTemperature_K);

		// upscaling leaf to canopy scale
		dRd *= 1.0;							// no need of upscaling becausa Rd is scaled by big-leaf Vcmax25

		//  calculate * in the absence of mitochondria respiraton (Pa) by Temperature
		//  (Eq.9; de Pury and Farquhar, 1997)
		dGanmaStar = 3.69 + 0.188 * (dTemperature_K - 298.15) + 0.0036 * pow((dTemperature_K - 298.15), 2);
	
		//  calculate PAR effectively absorbed by PSU
		//  (Eq.6; de Pury and Farquhar, 1997)
		dIle = dPPFD * (1.0 - c_f) / 2.0;

		//  calculate Temperature dependence of Jm
		//  (Eq.10; de Pury and Farquhar, 1997)
		dJm = m_Jmax25 * exp((dTemperature_K - 298.15) * 37000.0 / c_R / dTemperature_K / 298.15) 
			* (1.0 + exp((dS * 298.15 - dH)/ c_R / 298.15)) /
			(1.0 + exp((dS * dTemperature_K - dH) / c_R / dTemperature_K));

		//  calculate rate of electron transport rate per unit leaf area (umol m-2 s-1)
		//  (Eq.5; de Pury and Farquhar, 1997)
		dJ = (dIle + dJm - sqrt(pow(dIle + dJm, 2) - 4.0 * c_sl * dIle * dJm)) / 2.0 / c_sl;
	}
	if(m_iPhotosynthesisFunctions == 1){
		// Bernacchi et al. (2001 & 2003)
		// Bernacchi et al. 2001: Improved temperature response fuctions for models of Rubisco-limited photosynthesis
		//            Plant, Cell and Environment, 24, 253-259.
		// constant
		double dC_vl  = 26.35;
		double dHa_vl = 65.33;
		double dC_rd  = 18.72;
		double dHa_rd = 46.39;
		double dk25_kc = 1.0;
		double dC_kc  = 38.05;
		double dHa_kc = 79.43;
		double dk25_ko = 1.0;
		double dC_ko  = 20.30;
		double dHa_ko = 36.38;
		double dk25_gs = 1.0;
		double dC_gs  = 19.02;
		double dHa_gs = 37.83;

		// temperature responses basecd on Arrhenius Function
		dVCmax = Arrhenius2(m_Vcmax25, dC_vl, dHa_vl, dTemperature_K);
		dRd = Arrhenius2(0.0089 * m_Vcmax25, dC_rd, dHa_rd, dTemperature_K);
		dKc = Arrhenius2(dk25_kc, dC_kc, dHa_kc, dTemperature_K);
		dKo = Arrhenius2(dk25_ko, dC_ko, dHa_ko, dTemperature_K);
		dGanmaStar = Arrhenius2(dk25_gs, dC_gs, dHa_gs, dTemperature_K);

		// upscaling leaf to canopy scale
		dRd *= 1.0;							// no need of upscaling becausa Rd is scaled by big-leaf Vcmax25

		// ppm to Pa
		dKc = (dPressure * 100.0) * (dKc * pow(10, -6));
		dKo = (dPressure * 100.0) * (dKo * pow(10, -3));
		dGanmaStar = (dPressure * 100.0) * (dGanmaStar * pow(10, -6));

		// Bernacchi et al. 2001: In vivo temperature response functions of parameters required to model RuBP-limited photosynthesis
		//            Plant, Cell and Environment, 26, 1419-1430.
		double dC_jm  = 17.57;
		double dHa_jm = 43.54;
		dJm = Arrhenius2(m_Jmax25, dC_jm, dHa_jm, dTemperature_K);

		//  calculate rate of electron transport rate per unit leaf area (umol m-2 s-1)
		//  (Bernacchi et al., 2003)
		double dFaiII_bernacchi = 0.76 + 0.018 * (dTemperature_K - 273.15) - 3.7 * 0.0001 * (dTemperature_K - 273.15) * (dTemperature_K - 273.15);
		double dFaiII_max = 0.352 + 0.022 * (dTemperature_K - 273.15) - 3.4 * 0.0001 * (dTemperature_K - 273.15) * (dTemperature_K - 273.15);
		const double dBeta = 0.5;		// the fraction of absorbed quanta that reaches photosystem II

		dIle = dPPFD * dFaiII_max * (1.0 - c_f) * dBeta;

		dJ = (dIle + dJm - sqrt(pow(dIle + dJm, 2) - 4.0 * dFaiII_bernacchi * dIle * dJm)) / 2.0 / dFaiII_bernacchi;

	}
	if(m_iPhotosynthesisFunctions == 2){
		// von Caemmerer et al., 2009: Biochemical model of C3 photosynthesis
		//            In. Advances in Photosynthesis and RespirationVolume 29, 2009, pp 209-230 .
		// Table 9.1 in von Caemmerer 2009
		double dVl_E = 58.52 * 1000;		// J mol-1			(von Caemmerer et al., 2009)
		double dRd25 = m_Vcmax25 * 0.0089;	// umol m-2 s-1
		double dRd_E = 66.4 * 1000;			// J mol-1			(von Caemmerer et al., 2009)
		double dKc25 = 260.0 * 0.1;			// P (ubar * 0.1)	(von Caemmerer et al., 1994)
		double dKc_E = 59.36 * 1000;		// J mol-1			(Farquhar et al., 1980)
		double dKo25 = 179.0 * 100;			// P (mbar * 100)	(von Caemmerer et al., 1994)
		double dKo_E = 35.94 * 1000;		// J mol-1			(von Caemmerer et al., 2009)
		double dGs25 = 38.6 * 0.1;			// P (ubar * 0.1)	(von Caemmerer et al., 1994)
		double dGs_E = 23.4 * 1000;			// J mol-1			(von Caemmerer et al., 2009)
		double dJm_E = 37.0 * 1000;			// J mol-1			(von Caemmerer et al., 2009)
		double dH = 220.0 * 1000;			// J mol-1			(von Caemmerer et al., 2009)
		double dS = 710.0;					// J K-1 mol-1		(von Caemmerer et al., 2009)

		// temperature responses basecd on Arrhenius Function
		dVCmax = Arrhenius(m_Vcmax25, dVl_E, dTemperature_K);
		dRd    = Arrhenius(dRd25, dRd_E, dTemperature_K);
		dKc    = Arrhenius(dKc25, dKc_E, dTemperature_K);
		dKo    = Arrhenius(dKo25, dKo_E, dTemperature_K);
		dGanmaStar = Arrhenius(dGs25, dGs_E, dTemperature_K);

		// upscaling leaf to canopy scale
		dRd *= 1.0;							// no need of upscaling becausa Rd is scaled by big-leaf Vcmax25

		dJm = m_Jmax25 * exp( (dTemperature_K - 298.15) * dJm_E / 298.15 / c_R / dTemperature_K );
		dJm *= (1.0 + exp( (298.15 * dS - dH) / 298.15 / c_R ) );
		dJm /= (1.0 + exp( (dS * dTemperature_K - dH) / c_R / dTemperature_K ) );
			
		//  calculate PAR effectively absorbed by PSU
		//  (Eq.9.16; von Caemmerer et al., 2009)
		double dABS = 0.85;
		double dIle = dPPFD * dABS * (1.0 - c_f) / 2.0;

		//  calculate rate of electron transport rate per unit leaf area (umol m-2 s-1)
		//  (Eq.5; de Pury and Farquhar, 1997)
		dJ = (dIle + dJm - sqrt(pow(dIle + dJm, 2) - 4.0 * c_sl * dIle * dJm)) / 2.0 / c_sl;
	}
	if(m_iPhotosynthesisFunctions == 3){
		// Collatz et al., 1991: Physiological and environmental regulation of stomatal conductance, photosynthesis and transpiration: a model that includes a laminar boundary layer
		//            Agricultural and Forest Meteorology, 54, 107-136.
		double dVm_q10 = 2.4;
		double dRd_q10 = 2.0;
		double dKc25   = 30.0;			// Pa
		double dKc_q10 = 2.1;
		double dKo25   = 30000.0;		// Pa
		double dKo_q10 = 1.2;
		double dTau25  = 2600;
		double dTau_q10 = 0.57;
		double dJm_q10 = 2.4;

		// temperature responses basecd on Arrhenius Function
		dVCmax = Arrhenius3(m_Vcmax25, dVm_q10, dTemperature_K);
		dRd = Arrhenius3(m_Vcmax25 * 0.0089, dRd_q10, dTemperature_K);
		dKc = Arrhenius3(dKc25, dKc_q10, dTemperature_K);
		dKo = Arrhenius3(dKo25, dKo_q10, dTemperature_K);
		double dTau = Arrhenius3(dTau25, dTau_q10, dTemperature_K);
		dGanmaStar = c_Oi / 2.0 / dTau;		

		// upscaling leaf to canopy scale
		dRd *= 1.0;							// no need of upscaling becausa Rd is scaled by big-leaf Vcmax25

		// correcting high temperature stress
		double dA = 220000.0;			// J mol-1
		double dB = 703.0;				// J mo-1 K
		dVCmax = dVCmax * pow((1.0 + exp( (-dA + dB * dTemperature_K ) / c_R / dTemperature_K ) ), -1);
		dRd *= pow((1.0 + exp(1.3 * (dTemperature_K - 273.15 - 55.0) ) ), -1);

		dJm = Arrhenius3(m_Jmax25, dJm_q10, dTemperature_K);

		//  calculate PAR effectively absorbed by PSU
		//  (Eq.6; de Pury and Farquhar, 1997)
		dIle = dPPFD * (1.0 - c_f) / 2.0;

		//  calculate rate of electron transport rate per unit leaf area (umol m-2 s-1)
		//  (Eq.5; de Pury and Farquhar, 1997)
		dJ = (dIle + dJm - sqrt(pow(dIle + dJm, 2) - 4.0 * c_sl * dIle * dJm)) / 2.0 / c_sl;

	}
	if(m_iPhotosynthesisFunctions == 4){
		// Kosugi et al., 2003: Parameterization of the CO2 and H2O gas exchange of several temperate deciduous broad-leaved trees at the leaf scale considering seasoal changes
		//            Plant, Cell and Environment, 26, 285-301.
		// Table 1 in Kosugi et al. 2003
		double dVl_Ha = 67600.0;			// J mol-1			(Kosugi et al., 2003)
		double dVl_Hd = 204600.0;			// J mol-1			(Kosugi et al., 2003)
		double dVl_S  = 650.0;				// J K-1 mol-1		(Kosugi et al., 2003)
		double dRd25 = m_Vcmax25 * 0.0089;	// umol m-2 s-1
		double dRd_E = 66.405 * 1000;		// J mol-1			(Farquhar et al., 1980)
		double dKc25 = 27.5;				// P				(Harley et al., 1992)
		double dKc_E = 80500;				// J mol-1			(Harley et al., 1992)
		double dKo25 = 42000;				// P				(Harley et al., 1992)
		double dKo_E = 14500;				// J mol-1			(Harley et al., 1992)
		double dTau25 = 2321.0;				// (non-dimension)	(Harley et al., 1992)
		double dTau_E = -29000.0;			// J mol-1			(Harley et al., 1992)

		// temperature responses basecd on Arrhenius Function
		dVCmax = m_Vcmax25 * exp( (1.0 - 298.15 / dTemperature_K) * dVl_Ha / c_R / dTemperature_K );
		dVCmax /= (1.0 + exp( (dVl_S * dTemperature_K - dVl_Hd) / c_R / 298.15) );
		dRd = Arrhenius4(dRd25, dRd_E, dTemperature_K);
		dKc = Arrhenius4(dKc25, dKc_E, dTemperature_K);
		dKo = Arrhenius4(dKo25, dKo_E, dTemperature_K);
		double dTau = Arrhenius4(dTau25, dTau_E, dTemperature_K);
		dGanmaStar = c_Oi / 2.0 / dTau;

		dJm = dVCmax * (m_Jmax25 / m_Vcmax25);

		// upscaling leaf to canopy scale		
		dRd *= 1.0;							// no need of upscaling becausa Rd is scaled by big-leaf Vcmax25

		//  calculate PAR effectively absorbed by PSU
		//  (Eq.6; de Pury and Farquhar, 1997)
		dIle = dPPFD * (1.0 - c_f) / 2.0;

		//  calculate rate of electron transport rate per unit leaf area (umol m-2 s-1)
		//  (Eq.5; de Pury and Farquhar, 1997)
		dJ = (dIle + dJm - sqrt(pow(dIle + dJm, 2) - 4.0 * c_sl * dIle * dJm)) / 2.0 / c_sl;
	}
	if(m_iPhotosynthesisFunctions == 5){
		// Kattge & Knorr, 2007: Temperature acclimation in a biochemical model of photosynthesis: a reanalysis of data from 36 species
		//            Plant, Cell and Environment, 30, 1176-1190.
		// Medlyn et al., 2002: Temperature response of parameters of a biochemically based model of photosynthesis. II. A review of experimental data
		//            Plant, Cell and Environment, 25, 1167-1179.

		double dGrowthTemperature_C = dTg;

		// constant
		double dHa_vl = 71513.0;		// for Vcmax (J mol-1) (Kattge & Knorr, 2007)
		double dHd_vl = 200000.0;		// for Vcmax (J mol-1) (Kattge & Knorr, 2007)
		double dS_vl  = 668.39 - 1.07 * dGrowthTemperature_C;	// for Vcmax (J mol-1 C-1) (Kattge & Knorr, 2007)
		double dHa_j  = 49884.0;		// for Jmax (J mol-1) (Kattge & Knorr, 2007)
		double dHd_j  = 200000.0;		// for Jmax (J mol-1) (Kattge & Knorr, 2007)
		double dS_j   = 659.70 - 0.75 * dGrowthTemperature_C;	// for Jmax (J mol-1 C-1) (Kattge & Knorr, 2007)
		double dC_rd  = 18.72;			// (--)			(Bernacchi et al., 2001)
		double dHa_rd = 46.39;			// (kJ mol-1)	(Bernacchi et al., 2001)
		double dk25_kc = 404.9;			// (umol mol-1)	(Bernacchi et al., 2001)
		double dHa_kc = 79430.0;		// (J mol-1)	(Bernacchi et al., 2001)
		double dk25_ko = 278.4;			// (mol mol-1)	(Bernacchi et al., 2001)
		double dHa_ko = 36380.0;		// (J mol-1)	(Bernacchi et al., 2001)
		double dk25_gs = 42.75;			// (umol mol-1) (Bernacchi et al., 2001)
		double dHa_gs = 37830.0;		// (J mol-1)	(Bernacchi et al., 2001)

		// temperature responses basecd on Arrhenius Function
		dVCmax = m_Vcmax25 * exp(dHa_vl * (dTemperature_K - 298.15) / (298.15 * c_R * dTemperature_K ));
		dVCmax *= 1.0 + exp( (298.15 * dS_vl - dHd_vl) / c_R / 298.15 );
		dVCmax /= 1.0 + exp( (dTemperature_K * dS_vl - dHd_vl) / c_R / dTemperature_K );

		dJm = m_Jmax25 * exp(dHa_j * (dTemperature_K - 298.15) / (298.15 * c_R * dTemperature_K ));
		dJm *= 1.0 + exp( (298.15 * dS_j - dHd_j) / c_R / 298.15 );
		dJm /= 1.0 + exp( (dTemperature_K * dS_j - dHd_j) / c_R / dTemperature_K );

		dRd = Arrhenius2(0.01 * m_Vcmax25, dC_rd, dHa_rd, dTemperature_K);
		dKc = dk25_kc * exp(dHa_kc * (dTemperature_K - 298.15) / dTemperature_K / c_R / 298.15 );
		dKo = dk25_ko * exp(dHa_ko * (dTemperature_K - 298.15) / dTemperature_K / c_R / 298.15 );
		dGanmaStar = dk25_gs * exp(dHa_gs * (dTemperature_K - 298.15) / dTemperature_K / c_R / 298.15 );		

		// upscaling leaf to canopy scale
		dRd *= 1.0;							// no need of upscaling becausa Rd is scaled by big-leaf Vcmax25

		// ppm to Pa
		dKc = (dPressure * 100.0) * (dKc * pow(10, -6));
		dKo = (dPressure * 100.0) * (dKo * pow(10, -3));
		dGanmaStar = (dPressure * 100.0) * (dGanmaStar * pow(10, -6));

		//  calculate rate of electron transport rate per unit leaf area (umol m-2 s-1)
		dIle = dPPFD * 0.3;
		dJ = (dIle + dJm - sqrt(pow(dIle + dJm, 2) - 4.0 * 0.9 * dIle * dJm)) / 2.0 / 0.90;
	}
}

void CLeafPhotosynthesis::GanmaStar(double& dGanmaStar, const double& dTemperature_K, const double& dPressure)
{
	// dGanmaStar		: CO2 compensation point in absence of mitochondrial respiration (Pa)
	// calculate dGanmaStar for solving stomatal conductance model by Leuning (1995; PCE)

	// C4 photosynthesis
	if(m_iC3 == 0){
		// assmuption is made from SEIB-DGVM (Sato et al., 2015; JGR-B)
		dGanmaStar = 5.0;	// (ppm)
		dGanmaStar = dGanmaStar * (dPressure * 100.0) / pow(10, 6);
		return;
	}

	if(m_iPhotosynthesisFunctions == 0){
		// de Pury & Farquhar, 1997: Simple scaling of photosynthesis from leaves to canopies without the errors of big-leaf models
		//            Plant, Cell and Environment, 20, 537-557.
		dGanmaStar = 3.69 + 0.188 * (dTemperature_K - 298.15) + 0.0036 * pow((dTemperature_K - 298.15), 2);
	}
	if(m_iPhotosynthesisFunctions == 1){
		// Bernacchi et al. (2001 & 2003)
		// Bernacchi et al. 2001: Improved temperature response fuctions for models of Rubisco-limited photosynthesis
		//            Plant, Cell and Environment, 24, 253-259.
		// constant
		double dk25_gs = 1.0;
		double dC_gs  = 19.02;
		double dHa_gs = 37.83;
		dGanmaStar = Arrhenius2(dk25_gs, dC_gs, dHa_gs, dTemperature_K);

	}
	if(m_iPhotosynthesisFunctions == 2){
		// von Caemmerer et al., 2009: Biochemical model of C3 photosynthesis
		//            In. Advances in Photosynthesis and RespirationVolume 29, 2009, pp 209-230 .
		// Table 9.1 in von Caemmerer 2009
		double dGs25 = 38.6 * 0.1;			// P (ubar * 0.1)	(von Caemmerer et al., 1994)
		double dGs_E = 23.4 * 1000;			// J mol-1			(von Caemmerer et al., 2009)

		// temperature responses basecd on Arrhenius Function
		dGanmaStar = Arrhenius(dGs25, dGs_E, dTemperature_K);
	}
	if(m_iPhotosynthesisFunctions == 3){
		// Collatz et al., 1991: Physiological and environmental regulation of stomatal conductance, photosynthesis and transpiration: a model that includes a laminar boundary layer
		//            Agricultural and Forest Meteorology, 54, 107-136.
		double dTau25  = 2600;
		double dTau_q10 = 0.57;

		// temperature responses basecd on Arrhenius Function
		double dTau = Arrhenius3(dTau25, dTau_q10, dTemperature_K);
		dGanmaStar = c_Oi / 2.0 / dTau;	

	}
	if(m_iPhotosynthesisFunctions == 4){
		// Kosugi et al., 2003: Parameterization of the CO2 and H2O gas exchange of several temperate deciduous broad-leaved trees at the leaf scale considering seasoal changes
		//            Plant, Cell and Environment, 26, 285-301.
		// Table 1 in Kosugi et al. 2003
		double dTau25 = 2321.0;				// (non-dimension)	(Harley et al., 1992)
		double dTau_E = -29000.0;			// J mol-1			(Harley et al., 1992)

		// temperature responses basecd on Arrhenius Function
		double dTau = Arrhenius4(dTau25, dTau_E, dTemperature_K);
		dGanmaStar = c_Oi / 2.0 / dTau;
	}
	if(m_iPhotosynthesisFunctions == 5){
		// Kattge & Knorr, 2007: Temperature acclimation in a biochemical model of photosynthesis: a reanalysis of data from 36 species
		//            Plant, Cell and Environment, 30, 1176-1190.
		// Medlyn et al., 2002: Temperature response of parameters of a biochemically based model of photosynthesis. II. A review of experimental data
		//            Plant, Cell and Environment, 25, 1167-1179.
		double dHa_gs = 37830.0;		// (J mol-1)	(Bernacchi et al., 2001)
		double dk25_gs = 42.75;			// (umol mol-1) (Bernacchi et al., 2001)

		// temperature responses basecd on Arrhenius Function
		dGanmaStar = dk25_gs * exp(dHa_gs * (dTemperature_K - 298.15) / dTemperature_K / c_R / 298.15 );
	}
}

double CLeafPhotosynthesis::photosynthesis_model_C3(const double& dCO2, double& dVCmax, double& dRd, double& dKc, double& dKo, double& dGanmaStar, double& dJ, const double& dPressure)
{
	// dCO2				: CO2 concentration (ppm)
	// dVCmax			: photosynthetic Rubisco capacity per unit leaf area (mol m-2 s-1)
	// dRd				: dark respiration rate per unit leaf area (mol m-2 s-1)
	// dKc				: Michaelis-Menten constant of Rubisco for CO2 (Pa)
	// dKo				: Michaelis-Menten constant of Rubisco for O2 (Pa)
	// dGanmaStar		: CO2 compensation point in absence of mitochondrial respiration (Pa)
	// dJ				: rate of electron transport rate per unit leaf area (mol m-2 s-1)
	// dPressure		: atmospheric pressure (hPa)

	double dAn;			// net photosynthetic rate (umol m-2 s-1)

	// calculating CO2 pressure (Pa) from mixing ratio
	double dCp = (dPressure * 100.0) * (dCO2 * pow(10, -6));	// interceller CO2 pressure (Pa)

	// Effective Michaelis-Menten constant (Eq. 3; de Pury and Farquhar, 1997)
	double dK = dKc * (1.0 + c_Oi / dKo);

	// calculating Rubisco-limited photosythesis (Eq. 2; de Pury and Farquhar, 1997)
	double dAv = dVCmax * (dCp - dGanmaStar) / (dCp + dK);
	m_dAv = dAv - dRd;

	// calculating Electron-transport limited photosythesis (Eq. 2; de Pury and Farquhar, 1997)
	double dAj = dJ * (dCp - dGanmaStar) / 4.0 / (dCp + 2.0 * dGanmaStar);
	m_dAj = dAj - dRd;

	// co-limitation between Av and Aj 
	// Collatz et al., 1991 : Agricultural and Foreest Meteorology, 54, 107-136.
	// Physiological and environmental regulation of stomatal conductance, photosynthesis and transpiration: a model that includes a laminar boundary layer
	dAn = ( (dAv + dAj) - sqrt( pow( (dAv + dAj), 2) - 4.0 * c_dS * dAv * dAj) ) / 2.0 / c_dS - dRd;

	return dAn;
}

double CLeafPhotosynthesis::Goff_Gratch(const double& dT_degreeC, const double& dPressure)
{
	// calculate saturation water vapor given temperature using Goff-Gratch
	// dT        : temperature (degree C)
	// dPressure : atmospheric pressure (hPa)
	double dT;				// temperature (K)
	double dQsat;			// saturation water vapor (hPa)
	double vL;
	double T1 = 273.16;		//@triple point temperature of water (K)

	dT = dT_degreeC + 273.15;

	vL = 10.79574 * (1 - T1 / dT) - 5.028 * log10(dT / T1);
	vL = vL + 1.5475 * pow(10,-4) * (1 - pow(10,-8.2969 * (T1 / dT - 1)));
	vL = vL + 0.42873 * pow(10,-3) * (pow(10,4.76955 * (1 - T1 / dT) - 1)) + 0.78614;
	dQsat = 622.0 / dPressure * pow(10, vL);

	return dQsat;
}

double CLeafPhotosynthesis::calculate_air_density(const double& dT, const double& dP, const double& dRH, const int& bMol)
{
	// dT	: air temperature (degree C)
	// dP   : atmospheric pressure (hPa)
	// dRH	: relative humidity (%)
	// bMol	: 0 = kg m-3, 1 = mol m-3

	// saturation vapor pressure (hPa)
	double dSVP = Goff_Gratch(dT, dP);

	// vapor pressure (hPa)
	double dVP = dSVP * dRH / 100.0;

	// air density
	double dRow = 1.293 * dP / 1013.25 / (1.0 + dT / 273.15);	// kg m-3
	if(dRH != m_dErrorValue){
		dRow *= ( 1.0 - (0.378 * dVP / 1013.25) / (dP / 1013.25) );
	}

	if(bMol == 1){
		dRow *= 1000.0 / 28.96;
	}

	if(dT == m_dErrorValue || dP == m_dErrorValue){
		dRow = m_dErrorValue;
	}

	return dRow;
}

void CLeafPhotosynthesis::radiative_tansfer(const double& dPPFD_direct, const double& dPPFD_diffuse, const double& dLAI, 
	const double& dKb, const double& dKb2, const double& dRow_cb, const double& dRow_cd, const double& dKn, 
	double& dPPFD_sun, double& dPPFD_shade, double& dLAI_sun, double& dWeight_direct)
{
	double dPAR_direct_sun;		// absorbed beam PPFD  by sunlit leaf (umol m-2 s-1)
	double dPAR_diffuse_sun;    // absorbed diffuse PPFD by sunlit leaf (umol m-2 s-1)
	double dPAR_scatter_sun;    // absorbed scatter PPFD by sunlit leaf (umol m-2 s-1)
	double dPAR_diffuse_shade;  // absorbed diffuse PPFD by shaded leaf (umol m-2 s-1)
	double dPAR_scatter_shade;  // absorbed scatter PPFd by shaded leaf (umol m-2 s-1)
	double dPAR_ref;			// reflected PPFD by leaf (umol m-2 s-1)
	double dPAR_upward_sun;		// reflected PPFD into sunlit leaf by soil (umol m-2 s-1)
	double dPAR_upward_shade;	// reflected PPFD into shaded leaf by soil (umol m-2 s-1)

	// W1 : weighting factor of direct beam  (Eq. C6 in Wang and Leuning, 1998AFM)
	dWeight_direct = (1.0 - exp(-( dKb + dKn )            * dLAI ) ) / ( dKb + dKn );
	dWeight_direct = (1.0 - exp(-( dKb + dKn ) * m_dOmega * dLAI ) ) / ( dKb + dKn );

	// de Pury and Farquhar (1997) Eq. 22 devided by Eq. 15
	dWeight_direct = m_dOmega * ( 1.0 - exp(-dKn - dKb *  m_dOmega * dLAI) ) / (dKn + dKb *  m_dOmega * dLAI);
	dWeight_direct = ( 1.0 - exp(-dKn - dKb *  m_dOmega * dLAI) ) / (dKn + dKb *  m_dOmega * dLAI);
	dWeight_direct /= (1.0 - exp(-dKn) ) / dKn;

	// L1 : leaf area index for direct beam (Eq. C8 in Wang and Leuning, 1998AFM)
	if(dKb == 0.0){
		dLAI_sun = 0.0;
	}
	else{
		dLAI_sun = (1.0 - exp( -dKb * m_dOmega * dLAI) ) / dKb;
	}

	// direct PAR absorbed by the sunlit leaf (umol m-2 s-1) (Eq. 20b; de Pury and Farquhar, 1997PCE)
	dPAR_direct_sun = dPPFD_direct * ( 1.0 - m_dSigma_scattering ) * ( 1.0 - exp( -dKb * m_dOmega * dLAI ) );

	// diffuse PAR absorbed by the sunlit leaf (umol m-2 s-1) (Eq. 20c; de Pury and Farquhar, 1997PCE)
	dPAR_diffuse_sun = dPPFD_diffuse * ( 1.0 - dRow_cd ) * m_dOmega * ( 1.0 - exp( -( m_dKdd + dKb ) * m_dOmega * dLAI ) ) * m_dKdd / ( m_dKdd + dKb ) ;

	// scattered PAR absorbed by the sunlit leaf (umol m-2 s-1) (Eq. 20d; de Pury and Farquhar, 1997PCE)
	if(dKb == 0.0){
		dPAR_scatter_sun = 0.0;
	}
	else{
		dPAR_scatter_sun = dPPFD_direct * m_dOmega * ( ( 1.0 - dRow_cb ) * ( 1.0 - exp( -( dKb2 + dKb ) * m_dOmega * dLAI ) ) * dKb2 / ( dKb2 + dKb )-( 1.0 - m_dSigma_scattering ) * ( 1.0 - exp( -2.0 * dKb * m_dOmega * dLAI ) ) / 2.0);
	}

	// PAR absorbed by the sunlit leaf (umol m-2 s-1) (Eq. 20a; de Pury and Farquhar, 1997PCE)
	dPPFD_sun = dPAR_direct_sun + dPAR_diffuse_sun + dPAR_scatter_sun;
	
	// diffuse PAR absorbed by the shaded leaf (umol m-2 s-1) (Eq. A26b; de Pury and Farquhar, 1997PCE)
	dPAR_diffuse_shade = dPPFD_diffuse * ( 1.0 - dRow_cd ) * ( (1.0 - exp( -m_dKdd * m_dOmega * dLAI ) ) - ( 1.0 - exp( -( m_dKdd + dKb ) * m_dOmega * dLAI ) ) * m_dKdd * m_dOmega / ( m_dKdd + dKb ) );

	// scatter PAR absorbed by the shaded leaf (umol m-2 s-1) (Eq. A26c; de Pury and Farquhar, 1997PCE)
	if(dKb == 0.0){
		dPAR_scatter_shade = 0.0;
	}
	else{
		dPAR_scatter_shade = dPPFD_direct;
		dPAR_scatter_shade *= ( ( 1.0 - dRow_cb ) * (  (1.0 - exp( -dKb2 * m_dOmega * dLAI ) ) - ( 1.0 - exp( -( dKb2 + dKb ) * m_dOmega * dLAI ) ) * dKb2 * m_dOmega / ( dKb2 + dKb ) ) - ( 1.0 - m_dSigma_scattering ) * (  (1.0 - exp( -dKb * m_dOmega * dLAI ) ) - ( 1.0 - exp( -2.0 * dKb * m_dOmega * dLAI ) ) * m_dOmega / 2.0 ) ) ;
	}
	if(dPAR_scatter_shade < 0.0) dPAR_scatter_shade = 0.0;

	// reflected PAR (umol m-2 s-1)
	dPAR_ref = dRow_cb * dPPFD_direct * ( 1.0 - exp( -dKb2 * dLAI ) ) + dRow_cd * dPPFD_diffuse * ( 1.0 - exp( -m_dKdd            * dLAI ) );

	// PAR absorbed by the shaded leaf (umol m-2 s-1) (Eq. A26a; de Pury and Farquhar, 1997PCE)
	dPPFD_shade = dPAR_diffuse_shade + dPAR_scatter_shade;

	// absrobed PAR by sun leaf from that reflected PAR by soil (umol m-2 s-1) (Ryu et al., 2011GBC)
	dPAR_upward_sun = dPPFD_direct * ( 1.0 - m_dSigma_scattering ) + dPPFD_diffuse * ( 1.0 - dRow_cd ) - (dPPFD_sun + dPPFD_shade);
	dPAR_upward_sun *= m_dSoilReflectance_ppfd * exp( - dKb * m_dOmega * dLAI);

	dPPFD_sun += dPAR_upward_sun;

	// absrobed PAR by shade leaf from that reflected PAR by soil (umol m-2 s-1) (Ryu et al., 2011GBC)
	dPAR_upward_shade = dPPFD_direct * ( 1.0 - m_dSigma_scattering ) + dPPFD_diffuse * ( 1.0 - dRow_cd ) - (dPPFD_sun + dPPFD_shade);
	dPAR_upward_shade *= m_dSoilReflectance_ppfd * (1.0 - exp( - dKb * m_dOmega * dLAI) );
	dPPFD_shade += dPAR_upward_shade;
}

void CLeafPhotosynthesis::radiative_tansfer(double dPPFD_direct, double dPPFD_diffuse, const double& dPPFD_sun, const double& dPPFD_shade, const double& dLAI, 
	const double& dKb, const double& dKb2, const double& dRow_cb, const double& dRow_cd, const double& dKn,
	std::vector<double>& aryPAR, std::vector<double>& aryPAR_sun, std::vector<double>& aryPAR_shade, std::vector<double>& aryWeight_sun, std::vector<double>& aryVerticalFactor)
{
	// dPPFD_direct			: incoming beam PPFD at the top of the canopy (umol m-2 s-1)
	// dPPFD_diffuse		: incoming diffuse PPFD at the top of the canopy (umol m-2 s-1)
	// dPPFD_sun			: absorbed PPFD by sunlit leaf by whole canopy (umol m-2 s-1)
	// dPPFD_shade			: absorbed PPFD by shaded leaf by whole canopy (umol m-2 s-1)
	// dLAI					: leaf are index (m2 m-2)
	// dKb					: extinction coefficient of a canopy of black leaves for direct beam radiation (Wang and Leuning, 1998AFM)
	// dKb2					: extinction coefficient
	// dRow_cb              : relfectance of beam irradiance, uniform leaf-angle distribution (A19; de Pury and Farquhar, 1997PCE)
	// dRow_cd				: relfectance of diffuse irradiance, uniform leaf-angle distribution
	// dKb_day				: daily mean of extinction coefficient of a canopy of black leaves for direct beam radiation
	// aryPAR			    : absorbed PPFD by each layer (sunlit + shaded leaves) (umol m-2 s-1)
	// aryPAR_sun           : absorbed PPFD by sunlit leaf at each layer (umol m-2 s-1)
	// aryPAR_shaded        : absorbed PPFD by shaded leaf at each layer (umol m-2 s-1)
	// aryWeight_sun		: weighting factor in vcmax for sunlit leaf (dimensionless)
	// aryVerticalFactor    : ratio of vcmax in each layer relative to top of the canopy (from 0.0 to 1.0)

	double dPAR_sun;			// absorbed PPFD by sunlit leaf  (umol m-2 s-1)
	double dPAR_direct_sun;		// absorbed beam PPFD  by sunlit leaf (umol m-2 s-1)
	double dPAR_diffuse_sun;    // absorbed diffuse PPFD by sunlit leaf (umol m-2 s-1)
	double dPAR_scatter_sun;    // absorbed scatter PPFD by sunlit leaf (umol m-2 s-1)
	double dPAR_shade;			// absorbed PPFd by shaded leaf (umol m-2 s-1)
	double dPAR_diffuse_shade;  // absorbed diffuse PPFD by shaded leaf (umol m-2 s-1)
	double dPAR_scatter_shade;  // absorbed scatter PPFd by shaded leaf (umol m-2 s-1)
//	double dLAI_direct;
	double dWeight_sun;			// weighting factor in vcmax for sunlit leaf (dimensionless)

	vector<double> aryLAI;
	double dMultiLayerSplit = m_dMultiLayerSplit;	// LAI for each layer
	double dCurrentLAI;
	double dCummurativeLAI = 0.0;

	// radiative transfer for each canopy layer
	for(int iLayer = 0; iLayer < static_cast<int>(dLAI / m_dMultiLayerSplit) + 1 ; iLayer++){
		dCurrentLAI = dMultiLayerSplit;

		if(dLAI < dCummurativeLAI){
			dCurrentLAI = dMultiLayerSplit - (dCummurativeLAI - dLAI);
			dMultiLayerSplit = dLAI - dMultiLayerSplit * iLayer;
			dCummurativeLAI = dLAI;
		}
		aryLAI.push_back(dCurrentLAI);

		// sun-leaf PAR ***********************************************
		// direct PAR absorbed by the sunlit leaf (umol m-2 s-1) (Eq. 13; Leuning, 1995PCE after correcting clumping index)
		dPAR_direct_sun = dPPFD_direct * ( 1.0 - m_dSigma_scattering ) * dKb;

		// diffuse PAR absorbed by the sunlit leaf (umol m-2 s-1) (Eq. 20c; de Pury and Farquhar, 1997PCE)
		dPAR_diffuse_sun = dPPFD_diffuse * ( 1.0 - dRow_cd ) * m_dOmega * ( 1.0 - exp( -( m_dKdd + dKb ) * m_dOmega * dCurrentLAI ) ) * m_dKdd / ( m_dKdd + dKb ) ;

		// scattered PAR absorbed by the sunlit leaf (umol m-2 s-1) (Eq. 20d; de Pury and Farquhar, 1997PCE)
		if(dKb == 0.0){
			dPAR_scatter_sun = 0.0;
		}
		else{
			dPAR_scatter_sun = dPPFD_direct * m_dOmega * ( ( 1.0 - dRow_cb ) * ( 1.0 - exp( -( dKb2 + dKb ) * m_dOmega * dCurrentLAI ) ) * dKb2 / ( dKb2 + dKb )-( 1.0 - m_dSigma_scattering ) * ( 1.0 - exp( -2.0 * dKb * m_dOmega * dCurrentLAI ) ) / 2.0);
		}

		// PAR absorbed by the sunlit leaf (umol m-2 s-1) (Eq. 20a; de Pury and Farquhar, 1997PCE)
		dPAR_sun = dPAR_direct_sun + dPAR_diffuse_sun + dPAR_scatter_sun;
		aryPAR_sun.push_back(dPAR_sun);

		// weighting sunlit leaf parameters ******************************
		dWeight_sun = m_dOmega * exp( - dKb * m_dOmega * dCummurativeLAI);
		aryWeight_sun.push_back(dWeight_sun);

		// shade-leaf PAR ***********************************************
		// diffuse PAR absorbed by the shaded leaf (umol m-2 s-1) (Eq. A26b; de Pury and Farquhar, 1997PCE)
		// dPAR_diffuse_shade = dPPFD_diffuse * ( 1.0 - dRow_cd ) * ( (1.0 - exp( -m_dKdd * m_dOmega * dCurrentLAI ) ) - ( 1.0 - exp( -( m_dKdd + dKb ) * m_dOmega * dCurrentLAI ) ) * m_dKdd * m_dOmega / ( m_dKdd + dKb ) );
		dPAR_diffuse_shade = dPPFD_diffuse * ( 1.0 - dRow_cd ) * m_dKdd * exp( - m_dKdd * m_dOmega * dCummurativeLAI );		// Eq. A5 in de Pury and Farquhar 1997;
		dPAR_diffuse_shade *= (1.0 - dWeight_sun);

		// scatter PAR absorbed by the shaded leaf (umol m-2 s-1) (Eq. A26c; de Pury and Farquhar, 1997PCE)
		if(dKb == 0.0){
			dPAR_scatter_shade = 0.0;
		}
		else{
			dPAR_scatter_shade = dPPFD_direct;
			// dPAR_scatter_shade *= ( ( 1.0 - dRow_cb ) * (  (1.0 - exp( -dKb2 * m_dOmega * dCurrentLAI ) ) - ( 1.0 - exp( -( dKb2 + dKb ) * m_dOmega * dCurrentLAI ) ) * dKb2 * m_dOmega / ( dKb2 + dKb ) ) - ( 1.0 - m_dSigma_scattering ) * (  (1.0 - exp( -dKb * m_dOmega * dCurrentLAI ) ) - ( 1.0 - exp( -2.0 * dKb * m_dOmega * dCurrentLAI ) ) * m_dOmega / 2.0 ) ) ;
			dPAR_scatter_shade = ( 1.0 - dRow_cb ) * dKb2 * exp( -dKb2 * m_dOmega * dCummurativeLAI ) - (1.0 - m_dSigma_scattering) * dKb * exp( -dKb *  m_dOmega * dCurrentLAI );	// Eq. A8 in de Pury and Farquhar 1997;
			dPAR_scatter_shade = dPAR_scatter_shade * (1.0 - dWeight_sun);
		}
		if(dPAR_scatter_shade < 0.0) dPAR_scatter_shade = 0.0;

		// PAR absorbed by the shaded leaf (umol m-2 s-1) (Eq. A26a; de Pury and Farquhar, 1997PCE)
		dPAR_shade = dPAR_diffuse_shade + dPAR_scatter_shade;
		aryPAR_shade.push_back(dPAR_shade);

		// vertical gradient of parameters relative to canopy top
		aryVerticalFactor.push_back( exp( -dKn * ( dCummurativeLAI - dMultiLayerSplit )) );

		// total PAR absorved by each layer
		aryPAR.push_back(dPAR_sun * dWeight_sun * dCurrentLAI + dPAR_shade * dCurrentLAI );

		dPPFD_diffuse -= (dPAR_diffuse_sun + dPAR_diffuse_shade);
		if(dPPFD_diffuse < 0.0){
			dPPFD_diffuse = 0.0;
		}
		
		// increment LAI for next layers
		dCummurativeLAI += dCurrentLAI;
	}

	// correcting sun and shade leaf absorbed PPFD using total canopy APPFD
	// This multi-layer radiative transfer model did not consider the soil refection, but those for single layer did.
	dPAR_sun = 0.0, dPAR_shade = 0.0;
	for(iLayer = 0; iLayer < aryPAR_shade.size() ; iLayer++){
		dPAR_sun += aryPAR_sun[iLayer] * aryWeight_sun[iLayer] * aryLAI[iLayer];
		dPAR_shade += aryPAR_shade[iLayer] * aryLAI[iLayer];
	}
	if(dPAR_sun != 0)   dPAR_sun = dPPFD_sun / dPAR_sun;
	if(dPAR_shade != 0) dPAR_shade = dPPFD_shade / dPAR_shade;
	for(iLayer = 0; iLayer < aryPAR_shade.size() ; iLayer++){
		if(dPAR_sun != 0)   aryPAR_sun[iLayer] *= dPAR_sun;
		if(dPAR_shade != 0) aryPAR_shade[iLayer] *= dPAR_shade;
	}
}